#!/usr/bin/env python

'''
Start, and maintain, a 24/7 stream to the Twitch Stream API
'''

from __future__ import print_function
import argparse
import os
import logger
import sys
import ffmpeg
from dreamer import write_next_queue
from queue import Queue
from settings import SETTINGS


def cleanup_unfinished_stream(queue):
    '''
    Check and fix an unfinished stream process. If the last stream
    finished before removing the PLAYING value then the video was
    not finished. This will take that and update the provided queue
    to start from that position.
    '''
    unfinished_video, _ = ffmpeg.unfinished_video()
    if unfinished_video and queue.has_video(unfinished_video):
        finished_videos = queue.videos[:queue.videos.index(unfinished_video)]
        unfinished_videos = queue.videos[queue.videos.index(unfinished_video):]
        if len(finished_videos) > 0:
            logger.log("This video is in the queue so we better archive it "
                       "and create a new one", level=logger.WARNING)
            queue.update(videos=unfinished_videos)
            finished = Queue.write_queue(queue.sanitized_path(unfinished=True),
                                         finished_videos, queue)
            finished.archive()
            return queue
    return queue


def starting_queue(user_queue=None):
    '''
    Find, or create, the starting queue for this stream
    '''
    if user_queue is None:
        return Queue.first() or write_next_queue()
    return Queue(user_queue)


def config():
    '''
    Configure stream application from script args.
    '''
    parser = argparse.ArgumentParser()
    parser.add_argument('-l', '--list', help='queue to start with')
    return parser.parse_args()


if __name__ == '__main__':
    args = config()
    logger.basicConfig(
        SETTINGS['environment'],
        rollbar=SETTINGS['rollbar'],
        slack=SETTINGS['slack']
    )
    Queue.set_queue_path(os.path.join(os.getcwd(), 'content'))
    if ffmpeg.is_streaming():
        print('FATAL: An instance of ffmpeg is already running')
        sys.exit(1)
    try:
        queue = starting_queue(args.list)
        if queue is None:
            print('FATAL: Queue file provided is not valid')
            sys.exit(1)
        queue = cleanup_unfinished_stream(queue)
        ffmpeg.stream(
            queue,
            SETTINGS['stream']['key'],
            SETTINGS['stream']['channel'],
            SETTINGS['stream']['oauth_token']
        )
        logger.log('FFMPEG has stopped running', level=logger.ERROR)
    except Exception:
        logger.exception(sys.exc_info())
