import datetime
from functools import wraps


class throttle(object):
    '''
    Only call function once every X seconds
    '''

    def __init__(self, seconds=0):
        self.wait = datetime.timedelta(seconds=seconds)
        self.invocation_timestamps = {}

    def last_invocation(self, fn, new=None):
        '''
        Return timestamp of last method invocation
        '''
        if new is not None:
            self.invocation_timestamps[fn] = new
        return self.invocation_timestamps.get(
            fn, datetime.datetime.now() - self.wait)

    def __call__(self, fn):
        '''
        Allow each decorated method to be called, at most,
        one time per every `self.wait` seconds
        '''
        @wraps(fn)
        def wrapper(*args, **kwargs):
            last_invocation = self.last_invocation(fn)
            now = datetime.datetime.now()
            time_since_last_call = now - last_invocation
            if time_since_last_call > self.wait:
                self.last_invocation(fn, now)
                return fn(*args, **kwargs)
        return wrapper
