import os
import random
import subprocess
import re
import datetime
import ffprobe
from collections import defaultdict
from settings import SETTINGS

QUEUE_PATH = './content'


def memoize(f):
    '''
    Memoizes a function.
    TODO: Make callable in other classes
    '''
    memo = {}

    def helper(x):
        if x not in memo:
            memo[x] = f(x)
        return memo[x]
    return helper


class Video(object):

    def __init__(self, path):
        self.path = Video.sanitized_path(path)
        self.absolute_path = os.path.join(os.path.abspath(QUEUE_PATH), path)

    def __eq__(self, other):
        return type(self) is type(other) and self.path == other.path

    def __ne__(self, other):
        return type(self) is not type(other) or self.path != other.path

    @staticmethod
    def all(sort=False):
        '''
        Get all videos
        '''
        videos = []
        for root, dirs, files in os.walk(QUEUE_PATH, followlinks=True):
            path = os.path.relpath(root, QUEUE_PATH)
            for video in [f for f in files if f.endswith('.mp4')]:
                videos.append(Video(os.path.join(path, video)))
        if sort:
            return sorted(videos, key=lambda video: video.path)
        return videos

    @staticmethod
    def random(count=3, different_shows=True):
        '''
        Find shows with optimization to prevent shows with similar titles
        from appearing twice
        '''
        if different_shows is False:
            return random.sample(Video.all(), count)
        ignore_count = SETTINGS['minimum_shows_between_repeat']
        played_videos = Video.last_videos_played(ignore_count)
        last_played = played_videos[0]
        scheduled_videos = Video.scheduled_videos()
        videos = [v for v in Video.all()
                  if v.show.title not in SETTINGS['shows_to_ignore'] and
                  v not in played_videos and
                  v not in scheduled_videos and
                  v.show.family != last_played.show.family]
        shows_by_video = defaultdict(list)
        for v in videos:
            key = v.show.family
            shows_by_video[key].append(v)
        show_families = shows_by_video.values()
        chosen_families = random.sample(show_families, count)
        return [random.choice(videos) for videos in chosen_families]

    @staticmethod
    def last_videos_played(count):
        '''
        Get last number videos played
        '''
        from queue import Queue
        import ffmpeg
        # Get all played and playing queues
        queues = Queue.archived()
        current_queue = ffmpeg.current_queue()
        if current_queue:
            queues.append(current_queue)
        # get all videos played in the last count
        videos_played = []
        while queues and len(videos_played) < count:
            queue = queues.pop()
            videos = queue.videos[:]
            videos.reverse()
            videos_played += videos
            if len(videos_played) >= count:
                return videos_played[:count]
        return videos_played

    @staticmethod
    def scheduled_videos():
        '''
        Get all videos scheduled
        '''
        from queue import Queue
        return [v for q in Queue.all() for v in q.videos]

    @staticmethod
    def sanitized_path(path):
        '''
        Video names are problematic because they are never relative to the
        working directory as they are symlinked all over the file system.
        This method quickly sanitizes them so when we're talking about them
        we always expect them to be relative to `mp4s/path/to/video` regardless
        of where in the filesystem the actual video is.
        TODO: Make better use of os.path for this function when we have some
        sane default loctions on the server.
        '''
        return str(os.path.normpath('mp4s/%s' % path.split('mp4s')[-1]))

    @staticmethod
    @memoize
    def _duration(path):
        duration = ffprobe.duration(path)
        if duration:
            (hours, minutes, _seconds) = duration.split(':')
            seconds, microseconds = _seconds.split('.')
            return datetime.timedelta(hours=int(hours),
                                      minutes=int(minutes),
                                      seconds=int(seconds),
                                      microseconds=int(microseconds))
        return None

    @property
    def duration(self):
        '''
        Return a timedelta of the video's duration
        '''
        return Video._duration(self.absolute_path)

    @property
    def show(self):
        from show import Show
        return Show(path=os.path.dirname(self.path))

    @property
    def title(self):
        '''
        Human title
        '''
        title, ext = os.path.splitext(os.path.basename(self.path))
        return title.replace('_', ' ').title()
