defmodule FollowBeacon.StreamManager do
  use GenServer
  defstruct [streams: []]

  def start_link() do
    GenServer.start_link(__MODULE__, %FollowBeacon.StreamManager{})
  end

  def init(state) do
    schedule_scrape()
    streams = FollowBeacon.Scraper.get_all_streams()
    {:ok, %{state| streams: streams}}
  end

  def handle_info(:work, %FollowBeacon.StreamManager{streams: streams}) do
    new_streams = FollowBeacon.Scraper.get_all_streams()
    {offline, still_online} = detect_actual_diff(streams, new_streams)
    streams = MapSet.union(new_streams, still_online)
    IO.inspect({:offline, MapSet.size offline})
    IO.inspect({:online, MapSet.size still_online})
    # send_offline_notifications(offline)
    # send_online_notifications(online)
    schedule_scrape()
    {:noreply, %FollowBeacon.StreamManager{streams: new_streams}}
  end

  @doc """
    Needed to deal with the fact that the streams api might sometimes be wrong
    as a result of pagination not being done by a cursor
    TODO: Refactor to move flow into scraper
  """
  defp detect_actual_diff(old_streams, new_streams) do
    should_be_offline = MapSet.difference(new_streams, old_streams)
    still_online = should_be_offline
    |> Enum.chunk(100)
    |> Flow.from_enumerable(stages: 20, min_demand: 0, max_demand: 1)
    |> Flow.partition
    |> Flow.flat_map(&FollowBeacon.Scraper.get_streams_by_channel/1)
    |> Enum.to_list
    |> List.flatten
    |> Enum.into(%MapSet{})
    offline = MapSet.difference(should_be_offline, still_online)
    {offline, MapSet.difference(old_streams, new_streams)}
  end

  defp schedule_scrape() do
    Process.send_after(self(), :work, 1 * 60 * 1000) # In 2 hours
  end
end
