import $ from "jquery"
import React from "react"
import ReactDOM from "react-dom"
import update from 'react/lib/update';
import { debounce } from '../../utils/utils'

import getDndContext from './dnd-global-context';
import SequencerDnD from "./dnd_components/sequencer_dnd"
import PlayList from './dnd_components/play_list'

class Sequencer extends React.Component {
  constructor(props) {
    super(props);

    this.state = {
      assets: [],
      playListItems: [],
      deletes: [],
    }

    this.addToPlayList = this.addToPlayList.bind(this);
    this.movePlayListItem = this.movePlayListItem.bind(this);
    this.removeFromAssets = this.removeFromAssets.bind(this);
    this.dropOnPlayList = this.dropOnPlayList.bind(this);
    this._getAssetsBySource = debounce(this.getAssets.bind(this, "source_movie"), 1000);
    this._getAssetsByMetadata = debounce(this.getAssets.bind(this, "metadata"), 1000);
    this.submitPlayList = this.submitPlayList.bind(this);
    this.removeFromPlayList = this.removeFromPlayList.bind(this);
    this.moveAll = this.moveAll.bind(this);
    this.clearPlayList = this.clearPlayList.bind(this);
  }

  removeFromAssets(item) {
    const { assets } = this.state;
    const assetIndex = assets.findIndex(asset => asset.id === item.id)

    if(assetIndex > -1) {
      this.setState(update(this.state, {
        assets: {
          $splice: [
            [assetIndex, 1],
          ]
        }
      }))
    }
  }

  removeFromPlayList(item) {
    const { playListItems } = this.state;
    const playListIndex = playListItems.findIndex(asset => asset.id === item.id)

    if(playListIndex > -1) {
      this.setState(update(this.state, {
        playListItems: {
          $splice: [
            [playListIndex, 1],
          ]
        },
        deletes: {
          $push: [item]
        },
      }))
    }
  }

  addToPlayList(item) {
    if(!item) return;

    const { playListItems } = this.state;
    const assetIndex = playListItems.findIndex(asset => {
      return asset && item ? asset.id === item.id : false
    })

    if(assetIndex > -1) { return; }

    this.setState(update(this.state, {
      playListItems: {
        $push: [item]
      }
    }));
  }

  movePlayListItem(dragIndex, hoverIndex) {
    const { playListItems } = this.state;
    const dragItem = playListItems[dragIndex];

    this.setState(update(this.state, {
      playListItems: {
        $splice: [
          [dragIndex, 1],
          [hoverIndex, 0, dragItem],
        ],
      },
    }));
  }

  dropOnPlayList(item) {
    this.addToPlayList(item);
    this.removeFromAssets(item);
  }

  getAssets(queryBy, e) {
    let queryHash = {
      "source_movie": this.searchInput.value,
      "metadata": this.metafieldInput.value,
    }
    let query = queryHash[queryBy] || ""

    if(queryBy === "source_movie" && query.length < 1) return this.setState({assets: []});

    $.ajax({
      url: "/assets/sources",
      type: "GET",
      headers: {
        'Content-Type':'application/json',
        'Accept':'application/json'
      },
      data: { query, queryBy }
    }).then(data => {
      if(queryBy === "source_movie") {
        this.setState({assets: data})
      }
      if(queryBy == "metadata") {
        this.setState(update(this.state, {
          playListItems: { $unshift: data.reverse() },
          deletes: { $set: [] },
        }))
      }
    }, err => {
      console.error(err)
    })
  }

  submitPlayList(e) {
    e.preventDefault();
    let metafield = this.metafieldInput.value
    if(!confirm(`Submit Playlist for Metafield: ${metafield}`)) return;
    let assets = this.state.playListItems.filter(item => item)
    let deletes = this.state.deletes;
    let requestBody = { metafield, assets, deletes}

    $.ajax({
      url: "/assets/sequence",
      type: "POST",
      headers: {
        'Content-Type':'application/json',
        'Accept':'application/json'
      },
      data: JSON.stringify(requestBody)
    }).then(() => {},
      err => {
      console.error(err)
    })
  }

  moveAll() {
    const { assets, playListItems } = this.state
    let list = [...new Set(playListItems.concat(assets))]
    this.setState(update(this.state, {
      assets: { $set: [] },
      playListItems: { $set: list }
    }))
  }

  clearPlayList() {
    let metafield = this.metafieldInput.value
    if(metafield.length > 0 && !confirm(`Remove Assets from Metafield: ${metafield} ?`)) return;
    let assets = this.state.playListItems.filter(item => item)
    let deletes = [...new Set(this.state.deletes.concat(assets))]
    this.setState(update(this.state, {
      playListItems: { $set: [] },
      deletes: {$set: deletes },
    }))
  }

  render() {
    return (
      <div style={{display: 'flex'}}>
        <p>Asset Play Order</p>
        <div style={{ overflow: 'auto', clear: 'both', float: 'left', width: '50%', height: '50%' }}>
          Search Source Movies:
          <input type="text" ref={input => this.searchInput = input} onChange={this._getAssetsBySource}/>
          <button onClick={this.moveAll}>Move All</button>
          <SequencerDnD
            assets={this.state.assets}
            removeFromAssets={this.removeFromAssets} />
        </div>

        <div style={{ overflow: 'auto', clear: 'both', float: 'right', width: '50%', height: '50%' }}>
          Metafield:
          <form onSubmit={this.submitPlayList}>
            <input type="text" ref={input => this.metafieldInput = input} name="metafeld" onChange={this._getAssetsByMetadata} required/>
            <button type="submit">Submit</button>
          </form>
          <button onClick={this.clearPlayList}>Clear</button>
          <PlayList
            onDrop={this.dropOnPlayList}
            items={this.state.playListItems}
            movePlayListItem={this.movePlayListItem}
            removeFromPlayList={this.removeFromPlayList} />
        </div>
      </div>
    )
  }
}

export default getDndContext(Sequencer);
