defmodule BeanstalkGossipStrategy do
  import SweetXml, only: [sigil_x: 2]

  @moduledoc """
  This clustering strategy works by loading all instances with a given
  elastic beanstalk name. It will continually monitor and update it's
  connections every 5s.

  It assumes that all nodes share an app name are using ip addresses, and are unique
  based on their FQDN, rather than the base hostname.
  An example configuration is below:


      config :libcluster,
        topologies: [
          beanstalk_example: [
            strategy: #{__MODULE__},
            config: [
              app_name: "myapp",
              beanstalk_environments: ["myapp-prod"],
              polling_interval: 10_000]]]

  """
  use GenServer
  use Cluster.Strategy
  import Cluster.Logger

  alias Cluster.Strategy.State

  @default_polling_interval 10_000

  def start_link(opts), do: GenServer.start_link(__MODULE__, opts)

  def init(opts) do
    state = %State{
      topology: Keyword.fetch!(opts, :topology),
      connect: Keyword.fetch!(opts, :connect),
      disconnect: Keyword.fetch!(opts, :disconnect),
      config: Keyword.fetch!(opts, :config),
      meta: MapSet.new([])
    }

    {:ok, state, 0}
  end

  def handle_info(:timeout, state) do
    handle_info(:load, state)
  end

  def handle_info(
        :load,
        %State{topology: topology, connect: connect, disconnect: disconnect} = state
      ) do
    new_nodelist = MapSet.new(get_nodes(state))
    added = MapSet.difference(new_nodelist, state.meta)
    removed = MapSet.difference(state.meta, new_nodelist)

    new_nodelist =
      case Cluster.Strategy.disconnect_nodes(topology, disconnect, MapSet.to_list(removed)) do
        :ok ->
          new_nodelist

        {:error, bad_nodes} ->
          # Add back the nodes which should have been removed, but which couldn't be for some reason
          Enum.reduce(bad_nodes, new_nodelist, fn {n, _}, acc ->
            MapSet.put(acc, n)
          end)
      end

    new_nodelist =
      case Cluster.Strategy.connect_nodes(topology, connect, MapSet.to_list(added)) do
        :ok ->
          new_nodelist

        {:error, bad_nodes} ->
          bad_nodes
          # Remove the nodes which should have been added, but couldn't be for some reason
          Enum.reduce(bad_nodes, new_nodelist, fn {n, _}, acc ->
            MapSet.delete(acc, n)
          end)
      end

    Process.send_after(
      self(),
      :load,
      Keyword.get(state.config, :polling_interval, @default_polling_interval)
    )

    {:noreply, %{state | :meta => new_nodelist}}
  end

  def handle_info(_, state) do
    {:noreply, state}
  end

  def filters(envs) do
    ["Filter.1.Name": "tag:elasticbeanstalk:environment-name"] ++
      Enum.map(Enum.with_index(envs), fn {env, idx} -> {"Filter.1.Value.#{idx + 1}", env} end)
  end

  @spec get_nodes(State.t()) :: [atom()]
  defp get_nodes(%State{topology: topology, config: config}) do
    environments = Keyword.fetch!(config, :beanstalk_environments)
    app_name = Keyword.fetch!(config, :app_name)
    # environments = ["mission-control-prod", "prod-creative-aggregator-env"]
    ExAws.EC2.describe_instances(filters(environments))
    |> ExAws.request!()
    |> Map.get(:body)
    |> SweetXml.xpath(
         ~x"//DescribeInstancesResponse/reservationSet/item/instancesSet"l,
         ip: ~x"./item/privateIpAddress/text()"
       )
    |> Enum.filter(fn %{ip: ip} -> ip end)
    |> Enum.map(fn %{ip: ip} -> :"#{app_name}@#{ip}" end)
  end
end
