defmodule ChatBot do
  defstruct channel: nil, airing: nil, connection_handler: nil, login_handler: nil, messager: nil
  @twitch_stream_delay 12_000

  def start_chat_bot(opts), do: ChatBot.Supervisor.start_chat_bot(opts)

  def start_link({_, %{chat_bot_enabled: false}}) do
    {:ok, nil}
  end

  def start_link(args) do
    GenServer.start_link(__MODULE__, args)
  end

  def init({airing, channel}) do
    clients = make_clients(2)

    {:ok, connection_handler} = ChatBotConnectionHandler.start_link(clients, channel)
    {:ok, login_handler} = ChatBotLoginHandler.start_link(clients, ["\#" <> channel.login])
    {:ok, messager} = ChatBotMessager.start_link(clients, channel, airing.id)
    {:ok, %__MODULE__{channel: channel, airing: airing, messager: messager}}
  end

  def make_clients(n) do
    for _ <- 1..n do
      {:ok, client} = ExIrc.start_client!()
      client
    end
  end

  def handle_info(:send_auto_messages, %ChatBot{messager: messager, channel: chan} = state) do
    chan =
      MissionControlEx.Web.Channel.load_with_chat_data(chan)
      |> send_auto_messages(messager)

    {:noreply, state}
  end

  def send_auto_messages(%{chat_data: chat_data} = chan, _) when map_size(chat_data) == 0, do: nil

  def send_auto_messages(%{chat_commands: chat_commands} = chan, messager) do
    Enum.each(chat_commands, fn command ->
      if command.auto_send do
        case ChatBotMessager.generate_message(command, chan) do
          {:error, :no_fallback_message} ->
            nil

          message ->
            Process.send_after(messager, {:send_message, message}, @twitch_stream_delay)
            :timer.sleep(500)
        end
      end
    end)
  end

  def handle_info({:send_message, message}, %ChatBot{messager: messager} = state) do
    Process.send_after(messager, {:send_message, message}, 0)
    {:noreply, state}
  end
end
