defmodule ChatBotLoginHandler do
  require Logger

  @moduledoc """
  This is an example event handler that listens for login events and then
  joins the appropriate channels. We actually need this because we can't
  join channels until we've waited for login to complete. We could just
  attempt to sleep until login is complete, but that's just hacky. This
  as an event handler is a far more elegant solution.
  """
  def start_link(clients, channels) do
    GenServer.start_link(__MODULE__, [clients, channels, 0])
  end

  def init([clients, channels, 0]) do
    Enum.each(clients, fn client ->
      ExIrc.Client.add_handler(client, self())
    end)

    {:ok, {clients, channels, 0}}
  end

  def handle_info(:logged_in, {clients, channels, num_of_logged_ins_received})
      when num_of_logged_ins_received == length(clients) - 1 do
    Logger.info("All clients logged in, now connecting to channels", channel: hd(channels))

    Enum.map(clients, fn client ->
      channels |> Enum.map(&ExIrc.Client.join(client, &1))
    end)

    {:noreply, {clients, channels, num_of_logged_ins_received}}
  end

  def handle_info(:logged_in, {clients, channels, num_of_logged_ins_received}) do
    {:noreply, {clients, channels, num_of_logged_ins_received + 1}}
  end

  # Catch-all for messages you don't care about
  def handle_info(_msg, state) do
    {:noreply, state}
  end
end
