defmodule ChatBotMessager do
  defstruct clients: [],
            channel: nil,
            channel_name: nil,
            chat_command_strings: nil,
            chat_channel: nil,
            message_cache: nil,
            twitch_stream_id: nil

  require Logger
  alias MissionControlEx.Web.ChatCommand
  alias MissionControlEx.Web.Channel
  @delay_between_updates 30_000
  @twitch_stream_delay Application.get_env(:mission_control_ex, :twitch_delay)
  def start_link(clients, channel, twitch_stream_id) do
    GenServer.start_link(__MODULE__, %ChatBotMessager{
      clients: clients,
      channel_name: "\#" <> channel.login,
      channel: channel,
      twitch_stream_id: twitch_stream_id
    })
  end

  def init(%{clients: clients, channel: channel} = state) do
    {:ok, message_cache} =
      ConCache.start_link([ttl_check: :timer.seconds(1), ttl: :timer.seconds(10)], [])

    state = Map.put(state, :message_cache, message_cache)

    # Listen to one client's messages
    ExIrc.Client.add_handler(hd(clients), self())
    Process.send(self(), :update_channel_commands, [])
    :timer.send_interval(@delay_between_updates, :update_channel_commands)
    {:ok, state}
  end

  def handle_info(:update_channel_commands, %{channel: channel} = state) do
    channel = Channel.load_with_chat_data(channel.id)

    chat_command_strings =
      Enum.map(channel.chat_commands, fn chat_command -> chat_command.command end)

    {:noreply, %{state | channel: channel, chat_command_strings: chat_command_strings}}
  end

  # Received a message that may be a command
  def handle_info(
        {:received, "!" <> _ = text, user, channel} = msg,
        %{channel: %{chat_bot_enabled: true}, chat_command_strings: chat_command_strings} = state
      ) do
    if String.downcase(text) in chat_command_strings do
      process_chat_command(text, state)
    end

    {:noreply, state}
  end

  # TODO: Handle when message template key isn't in channel chat data
  defp process_chat_command(text, %{message_cache: cache, channel: channel}) do
    ConCache.get_or_store(cache, String.downcase(text), fn ->
      message =
        ChatCommand.get_by_command(String.downcase(text))
        |> generate_message(channel)

      case message do
        {:error, :no_fallback_message} ->
          Logger.error("[ChatBot] Unable to generate reply for command #{text}.
          Template: #{ChatCommand.get_by_command(String.downcase(text)).message_template}
          Current channel chat_data: #{inspect(channel.chat_data)}")

        _ ->
          Process.send(self(), {:send_message, message}, [])
      end
    end)
  end

  def generate_message(%{message_template: message_template} = command, channel) do
    try do
      if channel_has_data_for_template?(message_template, channel) do
        EEx.eval_string(message_template, data: channel.chat_data)
      else
        raise "Data not in channel"
      end
    rescue
      _ -> command.fallback_message || {:error, :no_fallback_message}
    end
  end

  def channel_has_data_for_template?(message_template, channel) do
    # Checks if any keys in template can't be resolved
    message_template
    |> split_template_into_string_keys
    |> Enum.reject(&valid_interpolation?(&1, channel.chat_data))
    |> Enum.empty?()
  end

  def split_template_into_string_keys(message_template) do
    message_template
    |> String.split("<%=")
    |> Enum.drop(1)
    |> Enum.map(fn string -> Enum.at(String.split(string, "%>"), 0) end)
  end

  # True if string_key is in chat_data, false otherwise
  def valid_interpolation?(string_key, chat_data) do
    case Code.eval_string(string_key, data: chat_data) do
      {nil, _} -> false
      _ -> true
    end
  end

  def handle_info({:send_message, message}, state) do
    Logger.info("Sending message...", channel: state.channel_name)
    {client, clients} = cycle_clients(state.clients)
    ExIrc.Client.msg(client, :privmsg, state.channel_name, message)
    {:noreply, %{state | clients: clients}}
  end

  # Discard all messages that aren't about logging into channels
  def handle_info(msg, state), do: {:noreply, state}

  def cycle_clients([h | t]), do: {h, t ++ [h]}
end
