defmodule ScheduleWorker do
  alias MissionControlEx.Web.{Schedule, StreamSchedule, ScheduleManager, Event, EventChunk}

  def load_event_list_from_csv(schedule, csv) do
    event_list =
      csv
      |> CSV.decode!(headers: true)
      |> Enum.to_list()
      |> Schedule.to_event_list()

    event_list = if schedule.shuffle, do: Enum.shuffle(event_list), else: event_list

    %{schedule | event_chunks: event_list, played_event_chunks: []}
    |> fill_queue
  end

  def fill_queue(%{event_chunks: [], played_event_chunks: []} = schedule),
    do: raise("No chunks to queue for schedule #{schedule.name}")

  def fill_queue(%{duration: nil} = schedule),
    do: %{
      schedule
      | queued_event_chunks: schedule.event_chunks,
        event_chunks: [],
        chunk_index: nil
    }

  def fill_queue(%{event_chunks: event_chunks, duration: duration} = schedule) do
    {chunks, schedule} = playable_chunks(schedule)

    num_chunks_to_queue =
      chunks
      |> Stream.transform(0, &halt_after_duration(&1, &2, duration))
      |> Enum.count()

    if not (num_chunks_to_queue > 0) do
      raise "Unable to queue chunks for schedule #{schedule.name} with duration #{
              schedule.duration
            }"
    end

    {queued_event_chunks, remaining_event_chunks} = Enum.split(chunks, num_chunks_to_queue)
    %{schedule | queued_event_chunks: queued_event_chunks, event_chunks: remaining_event_chunks}
  end

  def playable_chunks(
        %{repeat: true, event_chunks: event_chunks, played_event_chunks: played_event_chunks} =
          schedule
      ) do
    {event_chunks ++ Enum.reverse(played_event_chunks), %{schedule | played_event_chunks: []}}
  end

  def playable_chunks(%{event_chunks: event_chunks} = schedule), do: {event_chunks, schedule}

  def halt_after_duration(event_chunk, time, duration) do
    case time + EventChunk.duration(event_chunk) do
      time_after when time_after <= duration ->
        {[{event_chunk, time_after}], time_after}

      _ ->
        {:halt, time}
    end
  end

  def do_reset_stream_schedule(stream_schedule) do
    stream_schedule
    |> put_queued_chunks_back
    |> put_played_chunks_back
    |> fill_queue
  end

  def put_queued_chunks_back(
        %{queued_event_chunks: queued_event_chunks, event_chunks: event_chunks} = stream_schedule
      ),
      do: %{
        stream_schedule
        | queued_event_chunks: [],
          event_chunks: queued_event_chunks ++ event_chunks
      }

  def put_played_chunks_back(
        %{played_event_chunks: played_event_chunks, event_chunks: event_chunks} = stream_schedule
      ),
      do: %{
        stream_schedule
        | played_event_chunks: [],
          event_chunks: Enum.reverse(played_event_chunks) ++ event_chunks
      }

  def get_next_event_chunk(%{queued_event_chunks: []} = schedule),
    do: handle_empty_schedule(schedule)

  def get_next_event_chunk(%{chunk_index: nil} = schedule) do
    with new_schedule <- pop_queued_event_chunks(schedule),
         %{played_event_chunks: [next_chunk | _]} <- new_schedule do
      {new_schedule, EventChunk.link_schedule(next_chunk, new_schedule)}
    end
  end

  # Partial chunk case (for preview, resuming post-recovery, etc)
  def get_next_event_chunk(
        %{chunk_index: chunk_index, played_event_chunks: [%{events: events} = current_chunk | _]} =
          schedule
      ) do
    # Move to next if out of events on current chunk, return partial chunk otherwise
    case length(events) > chunk_index + 1 do
      false ->
        get_next_event_chunk(%{schedule | chunk_index: nil})

      true ->
        {_, unplayed_events} = Enum.split(events, chunk_index + 1)

        next_chunk =
          %{current_chunk | events: unplayed_events}
          |> EventChunk.link_schedule(schedule)

        {%{schedule | chunk_index: nil}, next_chunk}
    end
  end

  def pop_queued_event_chunks(
        %StreamSchedule{
          queued_event_chunks: [event_chunk | rest],
          played_event_chunks: played_event_list
        } = schedule
      ) do
    %StreamSchedule{
      schedule
      | queued_event_chunks: rest,
        played_event_chunks: [event_chunk | played_event_list]
    }
  end

  def handle_empty_schedule(%{event_chunks: remaining_chunks, stop_on_empty: true} = schedule)
      when length(remaining_chunks) > 0 do
    fill_queue(schedule) |> end_stream
  end

  # Empty cases
  ## API Schedule
  def handle_empty_schedule(%{api_url: api_url} = schedule) when api_url != nil do
    fill_event_list_from_api(schedule)
  end

  ## Stop if empty without repeat or api
  def handle_empty_schedule(%StreamSchedule{repeat: false} = schedule), do: end_stream(schedule)

  ## Repeat if repeat: true
  def handle_empty_schedule(%{repeat: true} = schedule), do: refill(schedule)

  def refill(%{stop_on_empty: false} = schedule), do: do_refill(schedule) |> get_next_event_chunk

  def refill(%{stop_on_empty: true} = schedule), do: do_refill(schedule) |> end_stream

  def do_refill(%{played_event_chunks: played_event_chunks, shuffle: shuffle} = schedule) do
    queued_event_chunks =
      if shuffle, do: Enum.shuffle(played_event_chunks), else: Enum.reverse(played_event_chunks)

    %{
      schedule
      | played_event_chunks: [],
        event_chunks: queued_event_chunks,
        chunk_index: nil
    }
    |> fill_queue
  end

  # def end_stream(schedule), do: {:error, :stream_done, schedule}
  def end_stream(schedule),
    do: {%{schedule | chunk_index: nil}, EventChunk.make_end_stream_chunk(schedule)}

  def fill_event_list_from_api(
        %{
          api_url: url,
          api_events_before_replay: unique_count,
          api_num_days_to_fetch: num_days,
          played_event_chunks: played_event_chunks
        } = schedule
      ) do
    amount_to_fetch = min(unique_count, 100)

    played_event_chunks = Enum.take(played_event_chunks, unique_count - amount_to_fetch)
    played_events = Enum.flat_map(played_event_chunks, fn %{events: events} -> events end)

    clip_chunks =
      MissionControlEx.Web.ApiStream.get_unique_clips(
        url,
        amount_to_fetch,
        num_days,
        played_events
      )
      |> Enum.map(&EventChunk.clip_to_chunk(&1, schedule))

    schedule =
      %{schedule | queued_event_chunks: clip_chunks, played_event_chunks: played_event_chunks}
      |> get_next_event_chunk
  end
end
