defmodule MissionControlEx.Web.AssetController do
  use MissionControlEx.Web, :controller

  alias MissionControlEx.Web.Asset
  @csv_thirty_mins "00:30:00"
  @generic_commerical "prod/mst3k/extras/BRB_105s/final/BRB_105s.ts"

  def index(conn, _params) do
    assets = Repo.all(Asset)
    render(conn, "index.html", assets: assets)
  end

  def new(conn, _params) do
    changeset = Asset.changeset(%Asset{})
    render(conn, "new.html", changeset: changeset)
  end

  def create(conn, %{"asset" => asset_params}) do
    changeset = Asset.changeset(%Asset{}, asset_params)

    case Repo.insert(changeset) do
      {:ok, _asset} ->
        conn
        |> put_flash(:info, "Asset created successfully.")
        |> redirect(to: asset_path(conn, :index))

      {:error, changeset} ->
        render(conn, "new.html", changeset: changeset)
    end
  end

  def show(conn, %{"id" => id}) do
    asset = Repo.get!(Asset, id)
    Transcode.s3_path(:transcode, asset)
    signed_url = Asset.sign_url(Transcode.s3_path(:transcode, asset))
    render(conn, "show.html", asset: asset, signed_url: signed_url)
  end

  def edit(conn, %{"id" => id}) do
    asset = Repo.get!(Asset, id)
    changeset = Asset.changeset(asset)
    render(conn, "edit.html", asset: asset, changeset: changeset)
  end

  def update(conn, %{"id" => id, "asset" => asset_params}) do
    asset = Repo.get!(Asset, id)
    changeset = Asset.changeset(asset, asset_params)

    case Repo.update(changeset) do
      {:ok, asset} ->
        conn
        |> put_flash(:info, "Asset updated successfully.")
        |> redirect(to: asset_path(conn, :show, asset))

      {:error, changeset} ->
        render(conn, "edit.html", asset: asset, changeset: changeset)
    end
  end

  def delete(conn, %{"id" => id}) do
    asset = Repo.get!(Asset, id)

    # Here we use delete! (with a bang) because we expect
    # it to always work (and if it does not, it will raise).
    Repo.delete!(asset)

    conn
    |> put_flash(:info, "Asset deleted successfully.")
    |> redirect(to: asset_path(conn, :index))
  end

  def search(conn, %{"q" => query}) do
    results =
      AssetSubtitleSearch.fetch(query)
      |> Enum.sort_by(&elem(&1, 1))
      |> Enum.reverse()

    render(conn, "asset_search.html", results: results, query: query)
  end

  def search(conn, %{}), do: redirect(conn, to: asset_path(conn, :index))

  def sequence(conn, _params) do
    render(conn, "asset_sequence.html")
  end

  def search_source(conn, %{"query" => query, "queryBy" => "source_movie"}) do
    results =
      Repo.all(Asset)
      |> Enum.reject(&(&1.source_movie == nil))
      |> Enum.filter(&String.contains?(&1.source_movie, query))
      |> Enum.sort_by(& &1.id)

    render(conn, :assets, assets: results)
  end

  def search_source(conn, %{"query" => query, "queryBy" => "metadata"}) do
    assets = Asset.get_assets_by_metafield(query)
    render(conn, :assets, assets: assets)
  end

  def export_csv(conn, %{"query" => query}) do
    conn
    |> put_resp_content_type("text/csv")
    |> put_resp_header("content-disposition", "attachment; filename=\"#{query}_assets.csv\"")
    |> send_resp(200, csv_content(query))
  end

  defp csv_content(query) do
    results = AssetSubtitleSearch.fetch(query)
    assets = Enum.map(results, &elem(&1, 0))
    headers = format_csv_headers(assets)

    results
    |> Enum.sort_by(&elem(&1, 1))
    |> Enum.reverse()
    |> Enum.map(&format_search_result(&1, headers))
    |> CSV.encode(headers: headers)
    |> Enum.to_list()
    |> format_headers_for_schedule
    |> to_string
  end

  defp format_search_result({asset, frequency}, headers) do
    additional_fields = %{
      frequency: frequency,
      schedule_block_length: @csv_thirty_mins,
      commercial_path: @generic_commerical
    }

    asset
    |> Map.merge(asset.metadata || %{})
    |> Map.merge(additional_fields)
    |> Map.take(headers)
  end

  defp format_csv_headers(assets) do
    metadata_headers =
      assets
      |> Enum.map(& &1.metadata)
      |> Utils.get_keys()

    [:schedule_block_length, :s3_path, :commercial_path, :frequency, :source_subtitles, :duration] ++
      metadata_headers
  end

  defp format_headers_for_schedule([headers | assets] = asset_list) do
    new_headers = String.replace(headers, ",s3_path,", ",asset_path,")
    [new_headers | assets]
  end

  def make_sequence(conn, %{"assets" => assets, "metafield" => metafield, "deletes" => deletes}) do
    with sequence <- format_sequence(assets, metafield, deletes),
         {:ok, results} <- Asset.update_metafields(sequence) do
      results = Map.values(results)
      ordered_results = Enum.sort_by(results, & &1.metadata[metafield])
      render(conn, :assets, assets: ordered_results)
    end
  end

  defp format_sequence(assets, metafield, deletes) do
    asset_structs =
      assets
      |> Enum.uniq_by(& &1["id"])
      |> Enum.map(&Poison.decode!(Poison.encode!(&1), as: %Asset{}))
      |> Enum.with_index()
      |> Enum.map(fn {asset, index} -> {asset, %{metafield => index}} end)

    delete_structs =
      deletes
      |> reject_deletes(assets)
      |> format_deletes(metafield)

    Enum.concat(asset_structs, delete_structs)
  end

  defp reject_deletes(deletes, assets) do
    Enum.reject(deletes, fn delete ->
      Enum.any?(assets, fn asset -> asset["id"] == delete["id"] end)
    end)
  end

  defp format_deletes(deletes, metafield) do
    deletes
    |> Enum.uniq_by(& &1["id"])
    |> Enum.map(&Poison.decode!(Poison.encode!(&1), as: %Asset{}))
    |> Enum.map(fn delete -> {delete, metafield, "__delete__"} end)
  end
end
