defmodule MissionControlEx.Web.ChannelController do
  use MissionControlEx.Web, :controller

  alias MissionControlEx.Web.Channel

  def index(conn, _params) do
    channels = Repo.all(Channel)
    render(conn, "index.html", channels: channels)
  end

  def new(conn, _params) do
    changeset =
      %Channel{}
      |> Repo.preload([:chat_commands])
      |> Channel.changeset()

    commands = Repo.all(MissionControlEx.Web.ChatCommand)

    render(conn, "new.html", changeset: changeset, commands: commands)
  end

  def create(conn, %{"channel" => channel_params}) do
    changeset = Channel.changeset(%Channel{}, channel_params)

    case Repo.insert(changeset) do
      {:ok, channel} ->
        conn
        |> put_flash(:info, "Channel created successfully.")
        |> redirect(to: channel_path(conn, :show, channel))

      {:error, changeset} ->
        render(conn, "new.html", changeset: changeset)
    end
  end

  def show(conn, %{"id" => id}) do
    channel = Channel.load(id)
    render(conn, "show.html", channel: channel)
  end

  def edit(conn, %{"id" => id}) do
    channel = Channel.load(id)
    changeset = Channel.changeset(channel)
    commands = Repo.all(MissionControlEx.Web.ChatCommand)
    render(conn, "edit.html", channel: channel, changeset: changeset, commands: commands)
  end

  def update(conn, %{"id" => id, "channel" => channel_params}) do
    channel = Channel.load(id)
    channel_params = Map.put_new(channel_params, "chat_commands", [])
    update_channel_commands(channel, channel_params)
    changeset = Channel.changeset(channel, channel_params)

    case Repo.update(changeset) do
      {:ok, channel} ->
        conn
        |> put_flash(:info, "Channel updated successfully.")
        |> redirect(to: channel_path(conn, :show, channel))

      {:error, changeset} ->
        render(conn, "edit.html", channel: channel, changeset: changeset)
    end
  end

  defp update_channel_commands(channel, %{"chat_commands" => chat_commands} = channel_params) do
    current_commands = Enum.map(channel.chat_commands, fn command -> command.command end)

    # Removal of all previous commands
    from(c in MissionControlEx.Web.ChannelCommand, where: c.channel_id == ^channel.id)
    |> Repo.delete_all()

    # Insertions of all new commands
    Enum.each(chat_commands, fn command_string ->
      chat_command = MissionControlEx.Web.ChatCommand.get_by_command(command_string)

      Repo.insert(%MissionControlEx.Web.ChannelCommand{
        channel_id: channel.id,
        chat_command_id: chat_command.id
      })
    end)
  end

  def delete(conn, %{"id" => id}) do
    channel = Repo.get!(Channel, id)

    # Here we use delete! (with a bang) because we expect
    # it to always work (and if it does not, it will raise).
    Repo.delete!(channel)

    conn
    |> put_flash(:info, "Channel deleted successfully.")
    |> redirect(to: channel_path(conn, :index))
  end
end
