defmodule MissionControlEx.Web.ScheduleManagerController do
  use MissionControlEx.Web, :controller
  alias MissionControlEx.Web.{StreamSchedule, ScheduleManager, Channel}

  def index(conn, _params) do
    schedule_managers =
      Repo.all(ScheduleManager)
      |> ScheduleManager.load_all()

    render(conn, "index.html", schedule_managers: schedule_managers)
  end

  def new(conn, %Ecto.Changeset{} = changeset), do: render_new(conn, changeset)
  def new(conn, _params), do: render_new(conn, ScheduleManager.changeset(%ScheduleManager{}))

  defp render_new(conn, changeset) do
    channels = Repo.all(Channel)

    stream_schedules =
      StreamSchedule.all_without_events()
      |> Enum.map(fn ss -> Map.drop(ss, [:events]) end)

    render(
      conn,
      "new.html",
      changeset: changeset,
      channels: channels,
      stream_schedules: stream_schedules,
      schedule_manager: %ScheduleManager{id: nil}
    )
  end

  def create(conn, %{"schedule_manager" => schedule_manager_params}) do
    with changeset <- ScheduleManager.changeset(%ScheduleManager{}, schedule_manager_params),
         {:ok, schedule_manager} <- Repo.insert(changeset),
         {:ok, schedule} <-
           StreamSchedule.create_default_schedule(schedule_manager, schedule_manager_params),
         schedule_manager_params <- format_create_params(schedule_manager_params, schedule),
         {:ok, _transactions} <-
           StreamSchedule.update_schedule_manager_assoc(schedule_manager, schedule_manager_params) do
      conn
      |> put_flash(:info, "ScheduleManager created successfully.")
      |> redirect(to: schedule_manager_path(conn, :index))
    else
      {:error, changeset} -> new(conn, changeset)
    end
  end

  defp format_create_params(params, nil), do: params

  defp format_create_params(params, default_schedule) do
    placeholder = %{"id" => default_schedule.id}

    params
    |> Map.put("default_schedule_id", default_schedule.id)
    |> Map.update(
         "schedules",
         %{placeholder: placeholder},
         &Map.put(&1, "placeholder", placeholder)
       )
  end

  def show(conn, %{"id" => id}) do
    schedule_manager =
      ScheduleManager.load(id)
      |> ScheduleManager.get_start_times(4)

    render(conn, "show.html", schedule_manager: schedule_manager)
  end

  def edit(conn, %{"id" => id}) do
    schedule_manager = ScheduleManager.load(id)
    channels = Repo.all(Channel)

    stream_schedules =
      StreamSchedule.all_without_events()
      |> Enum.map(fn ss -> Map.drop(ss, [:events]) end)

    changeset = ScheduleManager.changeset(schedule_manager)

    render(
      conn,
      "edit.html",
      schedule_manager: schedule_manager,
      changeset: changeset,
      channels: channels,
      stream_schedules: stream_schedules
    )
  end

  def update(conn, %{"id" => id, "schedule_manager" => schedule_manager_params}) do
    with schedule_manager <- Repo.get(ScheduleManager, id),
         changeset <- ScheduleManager.changeset(schedule_manager, schedule_manager_params),
         changeset <- ScheduleManager.validate_has_default(changeset, schedule_manager_params),
         {:ok, schedule_manager} <- Repo.update(changeset),
         {:ok, _transactions} <-
           StreamSchedule.update_schedule_manager_assoc(schedule_manager, schedule_manager_params) do
      conn
      |> put_flash(:info, "ScheduleManager updated successfully.")
      |> redirect(to: schedule_manager_path(conn, :show, schedule_manager))
    else
      {:error, changeset} ->
        render(
          conn,
          "edit.html",
          schedule_manager: %ScheduleManager{id: id},
          changeset: changeset,
          channels: [],
          stream_schedules: []
        )
    end
  end

  def delete(conn, %{"id" => id}) do
    schedule_manager = Repo.get!(ScheduleManager, id)
    Repo.delete!(schedule_manager)

    conn
    |> put_flash(:info, "ScheduleManager deleted successfully.")
    |> redirect(to: schedule_manager_path(conn, :index))
  end

  def get_next_chunk(conn, %{"id" => id} = params) do
    event_chunk =
      ScheduleManager.load(id)
      |> ScheduleManager.get_next_chunk(params["last_chunk_id"])

    json(conn, event_chunk)
  end

  def preview(conn, %{"schedule_manager_id" => manager_id, "length" => preview_length} = params) do
    {id, _} = Integer.parse(manager_id)
    {preview_length, _} = Integer.parse(preview_length)

    schedule_manager =
      ScheduleManager.load(id)
      |> get_preview(preview_length)

    render(conn, :preview, schedule_manager: schedule_manager)
  end

  defp get_preview(%{stream_schedules: []} = schedule_manager, preview_length),
    do: schedule_manager

  defp get_preview(%{status: status} = schedule_manager, preview_length)
       when status in ["waiting", "streaming", "recovering"] do
    schedule_manager
    |> ScheduleManager.generate_event_stream()
    |> Stream.take(preview_length)
    |> Enum.at(-1)
    |> Map.get(:schedule_manager)
  end

  defp get_preview(schedule_manager, _), do: schedule_manager

  def launch(conn, %{"id" => id}) do
    with schedule_manager <- ScheduleManager.load(id),
         true <- ScheduleManager.is_valid?(schedule_manager) && schedule_manager.status == nil,
         changeset <- ScheduleManager.changeset(schedule_manager, %{status: "waiting"}),
         {:ok, schedule_manager} <- Repo.update(changeset) do
      conn
      |> put_flash(:info, "Success! Schedule Manager put in status 'waiting' for streaming")
      |> redirect(to: schedule_manager_path(conn, :index))
    else
      false ->
        conn
        |> put_flash(:error, "Error: Schedule Manager already launched and/or has no channel")
        |> redirect(to: schedule_manager_path(conn, :index))
    end
  end
end
