defmodule MissionControlEx.Web.StreamScheduleController do
  use MissionControlEx.Web, :controller

  alias MissionControlEx.Web.{
    StreamSchedule,
    Event,
    EventChunk,
    Feed,
    Asset,
    Schedule,
    ScheduleManager,
    Schedule.ScheduleInvalidError
  }

  require Utils

  action_fallback(MissionControlEx.Web.FallbackController)

  def index(conn, _params) do
    stream_schedules =
      Repo.all(StreamSchedule)
      |> Enum.map(&StreamSchedule.load/1)

    render(conn, :index, stream_schedules: stream_schedules)
  end

  def new(conn, _params) do
    changeset = StreamSchedule.changeset(%StreamSchedule{})
    render(conn, "new.html", changeset: changeset)
  end

  def create(conn, %{"stream_schedule" => stream_schedule_params}) do
    changeset = StreamSchedule.changeset(%StreamSchedule{}, stream_schedule_params)

    case Repo.insert(changeset) do
      {:ok, _stream_schedule} ->
        conn
        |> put_flash(:info, "Twitch stream created successfully.")
        |> redirect(to: stream_schedule_path(conn, :index))

      {:error, changeset} ->
        render(conn, "new.html", changeset: changeset)
    end
  end

  def show(conn, %{"id" => id} = params) do
    stream_schedule = StreamSchedule.load(id)
    changeset = StreamSchedule.changeset(stream_schedule)
    render(conn, :show, stream_schedule: stream_schedule, changeset: changeset)
  end

  def edit(conn, %{"id" => id, "stream_schedule" => ss_params} = params) do
    stream_schedule = StreamSchedule.load(id)
    changeset = StreamSchedule.changeset(stream_schedule, ss_params)
    render(conn, "edit.html", stream_schedule: stream_schedule, changeset: changeset)
  end

  def update(conn, %{"id" => id, "csv_schedule" => schedule} = params) do
    with ss_params <- null_to_nil(params),
         changeset <- StreamSchedule.changeset(StreamSchedule.load(id), ss_params),
         stream_schedule <- Ecto.Changeset.apply_changes(changeset),
         {:ok, preview_schedule} <- load_stream_schedule_with_csv(stream_schedule, schedule),
         final_changeset <- StreamSchedule.changeset(changeset, Map.from_struct(preview_schedule)),
         preview_length <- ss_params["preview_length"] || "100",
         {preview_length, _} <- Integer.parse(preview_length),
         do: do_update(conn, preview_schedule, final_changeset, preview_length, params)
  end

  def save_curated_changes(
        conn,
        %{"id" => id, "queued_event_chunks" => queued_event_chunks} = params
      ) do
    ss = StreamSchedule.load(id)

    queued_event_chunks =
      queued_event_chunks
      |> Poison.decode!()
      |> Enum.map(&EventChunk.make_event_chunk_from_json(&1, id))

    StreamSchedule.persist_changes(%{ss | queued_event_chunks: queued_event_chunks})
    show(conn, params)
  end

  def curator(conn, %{"id" => id} = params) do
    stream_schedule = StreamSchedule.load(id)
    changeset = StreamSchedule.changeset(stream_schedule)
    render(conn, :curator, stream_schedule: stream_schedule, changeset: changeset)
  end

  def create_curated_chunk(conn, %{"id" => id, "url" => url} = params) do
    with %{} = data <- TwitchApiUtils.get_json(url) do
      event =
        Event.play_curated_asset(id, "curated_online", url, %{
          duration: data["duration"]
        })

      chunk = EventChunk.make_event_chunk([event], data)
      json(conn, chunk)
    else
      {:error, _} -> {:curator_url_error, "blah"}
    end
  end

  def do_update(
        conn,
        preview_schedule,
        changeset,
        preview_length,
        %{"persist" => "false"} = params
      ) do
    with {:ok, events} <- get_preview_events(preview_schedule, preview_length),
         do:
           render(
             conn,
             :show,
             stream_schedule: preview_schedule,
             changeset: changeset,
             events: events
           )
  end

  def do_update(
        conn,
        _preview_schedule,
        final_changeset,
        preview_length,
        %{"persist" => "true"} = params
      ) do
    with {:ok, updated_ss} <- Repo.update(final_changeset),
         {:ok, events} <- get_preview_events(updated_ss, preview_length),
         do:
           render(
             conn,
             :show,
             stream_schedule: updated_ss,
             changeset: final_changeset,
             events: events,
             saved: "true"
           )
  end

  defp load_stream_schedule_with_csv(stream_schedule, %Plug.Upload{
         content_type: "text/csv",
         path: csv
       }) do
    try do
      schedule = StreamSchedule.load_event_list_from_csv(stream_schedule, File.stream!(csv))
      {:ok, schedule}
    rescue
      err in ScheduleInvalidError ->
        Utils.rollbar_report_warning(err, %{stream_schedule: stream_schedule})
        {:invalid_csv_schedule, err}
    end
  end

  defp load_stream_schedule_with_csv(stream_schedule, _), do: {:ok, stream_schedule}

  defp get_preview_events(stream_schedule, preview_length) do
    default_manager = ScheduleManager.create_empty_manager(stream_schedule)

    try do
      events =
        stream_schedule
        |> Map.get(:schedule_manager_id)
        |> (fn id -> id || default_manager end).()
        |> ScheduleManager.load()
        |> ScheduleManager.with_stream_schedule(%{stream_schedule | default_schedule: true})
        |> ScheduleManager.generate_event_stream()
        |> Stream.take(preview_length)
        |> Enum.at(-1)
        |> Map.get(:schedule_manager)
        |> Map.get(:events)

      {:ok, events}
    rescue
      error -> {:preview_error, error}
    end
  end

  def delete(conn, %{"id" => id}) do
    stream_schedule = Repo.get!(StreamSchedule, id)

    # Here we use delete! (with a bang) because we expect
    # it to always work (and if it does not, it will raise).
    Repo.delete!(stream_schedule)

    conn
    |> put_flash(:info, "Twitch stream deleted successfully.")
    |> redirect(to: stream_schedule_path(conn, :index))
  end

  def reset_stream(conn, %{"id" => id}) do
    Repo.get!(StreamSchedule, id)
    |> StreamSchedule.reset_stream()

    conn
    |> put_flash(:info, "Twitch stream reset successfully.")
    |> redirect(to: stream_schedule_path(conn, :index))
  end

  def duplicate(conn, %{"id" => id}) do
    schedule =
      StreamSchedule.load(id)
      |> StreamSchedule.duplicate()

    changeset = StreamSchedule.changeset(schedule)

    conn
    |> put_flash(
         :info,
         "Created duplicate schedule named #{schedule.name}, don't forget to make a new schedule_manager"
       )
    |> redirect(to: stream_schedule_path(conn, :index))
  end

  def editor(conn, %{"id" => id} = _params),
    do: render(conn, "schedule_edit.html", stream_schedule: StreamSchedule.load(id))

  def assets(conn, %{"id" => id} = params),
    do:
      render(conn, :ts_assets, %{
        stream_schedule: StreamSchedule.load(id),
        assets: Asset.all_order(),
        feeds: Feed.all()
      })

  defp null_to_nil(%{} = params), do: Enum.reduce(params, params, &to_nil/2)
  defp to_nil({key, "null"}, map), do: Map.put(map, key, nil)
  defp to_nil({key, val}, map), do: map
end
