defmodule GenerateTranscodeJobs do
  use MissionControlEx.Web, :model
  alias MissionControlEx.Web.{Repo, Asset, Manifest}
  require Utils
  @bucket "twitch-creative-video-repository"
  # @spec run(String.t) :: [{String.t, String.t}]

  def run(stream_source, manifest \\ %{}) do
    with rows <- CSV.decode(stream_source, headers: true, strip_fields: true),
         {:ok, structs} <- rows_to_structs(rows, manifest) do
      upsert_all_rows(structs, manifest)
    else
      {:error, error} -> {:error, error}
    end
  end

  def rows_to_structs(rows, manifest) do
    structs = Enum.map(rows, &to_struct(&1, manifest))
    valid? = Enum.find(structs, :ok, &is_error?/1)

    case valid? do
      :ok -> {:ok, structs}
      {:error, error} -> {:error, error}
    end
  end

  defp is_error?({:error, _error}), do: true
  defp is_error?(_), do: false

  def from_file(file_path, manifest \\ %{}) do
    File.stream!(file_path)
    |> run(manifest)
  end

  def upsert_all_rows(rows, %Manifest{} = manifest) do
    Repo.transaction(fn ->
      Manifest.remove_old_assocs(manifest, rows)
      Repo.insert_all(Asset, rows, on_conflict: :nothing)
      Manifest.add_assocs(manifest, rows)
    end)
  end

  def upsert_all_rows(_, _), do: nil

  defp to_struct({:error, error}, _manifest, _exclude), do: {:error, error}

  defp to_struct({:ok, row}, manifest, exclude \\ []) do
    struct = struct(Asset) |> Map.put(:s3_path, final_s3_path(manifest, row["source_movie"]))
    ignored_fields = [:id, :inserted_at, :updated_at] ++ exclude
    asset_fields = Asset.__schema__(:fields) -- ignored_fields

    row
    |> Map.put_new("commercial_breaks_ms", "")
    |> parse_ad_blocks
    |> Enum.reduce(struct, fn {key, val}, acc ->
         case parse_key_val(key, val) do
           :error ->
             acc

           nil ->
             acc

           v ->
             case String.to_atom(key) in asset_fields do
               true -> Map.put(acc, String.to_atom(key), v)
               false -> add_metadata(acc, key, v)
             end
         end
       end)
    |> Map.from_struct()
    |> Map.take(asset_fields)
  end

  defp parse_key_val("s3_path", ""), do: :error
  defp parse_key_val("trimmings", ""), do: ""
  defp parse_key_val("commercial_breaks_ms", ""), do: []
  defp parse_key_val("commercial_breaks_ms", v) when is_list(v), do: v
  defp parse_key_val(k, ""), do: nil

  defp parse_key_val(k, v) when k in ["duration", "raw_duration", "commercial_breaks_ms"],
    do: Poison.decode!(v)

  defp parse_key_val(k, v), do: v

  defp add_metadata(acc, key, v) when v in [nil, ""], do: acc
  defp add_metadata(acc, key, v), do: Map.put(acc, :metadata, Map.put(acc.metadata, key, v))

  def final_s3_path(manifest, file_name) do
    manifest_name = Map.get(manifest, :name) || Map.get(manifest, "name") || "generic"
    final_name = Path.basename(file_name) |> Path.rootname()
    "#{Mix.env()}/#{manifest_name}/#{final_name}/final/#{final_name}.ts"
  end

  defp group_s3_by_file_types(files), do: Enum.group_by(files, &Utils.get_type(&1.key))

  defp parse_ad_blocks(%{"commercial_breaks_ms" => ""} = row) do
    parsed_ad_blocks =
      Enum.reduce(row, [], fn {key, val}, acc ->
        case is_commercial_header?(key, val) do
          true -> acc ++ [Utils.timestamp_to_ms(val)]
          _ -> acc
        end
      end)

    Map.put(row, "commercial_breaks_ms", Enum.sort(parsed_ad_blocks))
  end

  defp parse_ad_blocks(row), do: row

  def is_commercial_header?(header, ""), do: false
  def is_commercial_header?("commercial break timestamp" <> _rest, val), do: true
  def is_commercial_header?(header, val), do: false

  def create_csv(params, manifest_params) do
    params
    |> Enum.to_list()
    |> Enum.sort_by(fn {k, _v} -> Poison.decode!(k) end)
    |> Enum.map(fn {_k, v} -> {:ok, v} end)
    |> Enum.map(&to_struct(&1, manifest_params, [:metadata]))
    |> CSV.encode(headers: true)
  end
end
