defmodule Transcode do
  require Utils
  require RevCaptions
  require Logger
  require Transcode.Trimmer
  alias MissionControlEx.Web.{Asset}

  @one_second 1_000
  @five_gb 5 * 1_000_000_000
  @five_mb 5 * 1024 * 1024
  @min_byte_size 10
  @video_bucket "twitch-creative-video-repository"
  @exaws_config ExAws.Config.new(:s3, %{
                  access_key_id: "AKIAJ4RHGC5TBNDDSR3A",
                  secret_access_key: "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"
                })
  @media_path Application.get_env(:mission_control_ex, :media_path)

  use AsyncWith
  use Retry
  # 5 hours
  @async_with_timeout 18_000_000

  def process_asset(_, retries) when retries > 3, do: nil

  def process_asset(%Asset{} = asset, retries \\ 0) do
    with {:ok, raw_asset_path} <- check_raw_asset(asset),
         local_raw_path <- download_s3(raw_asset_path, asset_extname(:raw, asset), asset),
         {:ok, local_transcode_path} <- transcode_asset(local_raw_path, asset),
         {:ok, transcode_uploaded} <-
           upload_file(s3_path(:transcode, asset), local_transcode_path),
         {:ok, asset_transcoded} <- update_asset("transcode", asset, transcode_uploaded),
         {:ok, asset_with_duration} <-
           get_transcode_duration(local_transcode_path, asset_transcoded),
         {:ok, subtitle_status} <- check_subtitles(asset),
         {:ok, order} <- order_captions(subtitle_status, asset_with_duration),
         {:ok, asset_caption_ordered} <- update_asset("waiting_on_captions", asset, order),
         {:ok, order_status} <- check_order_status(order, asset),
         {:ok, local_subtitle_path} <- validate_captions(order_status, asset),
         {:ok, local_captioned_path} <-
           apply_captions(local_subtitle_path, local_transcode_path, asset),
         {:ok, captioned_uploaded} <-
           upload_file(s3_path(:captioned, asset), local_captioned_path),
         {:ok, asset_captioned} <- update_asset("captioned", asset, captioned_uploaded),
         {:ok, local_ts_path} <- ts_convert(local_captioned_path, asset),
         {:ok, ts_uploaded} <- upload_file(s3_path(:ts, asset), local_ts_path),
         {:ok, asset_ts_converted} <- update_asset("ts_convert", asset, ts_uploaded),
         {:ok, local_trim_path} <- trim_asset(local_ts_path, asset),
         {:ok, trim_uploaded} <- upload_file(s3_path(:trim, asset), local_trim_path),
         {:ok, asset_trimmed} <- update_asset("trim", asset, trim_uploaded),
         {:ok, copy_trim_path} <-
           copy_to_(s3_path(:trim, asset_trimmed), s3_path(:final, asset), asset),
         {:ok, asset_validated} <- final_validation(asset_trimmed, copy_trim_path) do
      complete(asset_validated)
    else
      {:subtitle_order, _} -> nil
      {:error, :retry} -> process_asset(asset, retries + 1)
    end
  end

  def start_link(asset) do
    {:ok, transcode_child} =
      Task.start_link(fn ->
        try do
          process_asset(asset)
        rescue
          err in AsyncWith.ClauseError -> log_error(err.term, asset)
        after
          {:ok, files} = File.rm_rf("#{@media_path}/#{asset.id}")
        end
      end)

    Task.Supervisor.start_child(MissionControlEx.Web.OwnershipSupervisor, fn ->
      Utils.wait_for_finish(transcode_child, fn ->
        Asset.touch(asset)
      end)
    end)

    {:ok, transcode_child}
  end

  @doc """
  Check Raw Asset
  """
  def check_raw_asset(asset) do
    Logger.info("(Asset ID: #{asset.id}) Check raw Asset")

    case s3_path_exists?(s3_path(:raw, asset)) do
      :ok ->
        {:ok, s3_path(:raw, asset)}

      :error ->
        {:ok, _} = convert_and_copy_subtitles(asset, AssetSubtitleSearch.subtitle_s3_path(asset))
        {:ok, _} = copy_to_(asset.source_subtitles, s3_path(:subtitles, asset), asset)
        {:ok, _} = copy_to_(asset.source_movie, s3_path(:raw, asset), asset)
        {:error, :retry}
    end
  end

  def convert_and_copy_subtitles(%{source_subtitles: subtitles}, _)
      when subtitles in ["None_Required", "order_captions"],
      do: {:ok, nil}

  def convert_and_copy_subtitles(%{source_subtitles: source_subtitles} = asset, subtitle_dest) do
    local_subtitle_path =
      download_s3(source_subtitles, Utils.get_extname(source_subtitles), asset)

    local_srt_path = create_media_file(".srt", asset)
    :ok = run_shell("ffmpeg -y -i #{local_subtitle_path} #{local_srt_path}")
    upload_file(subtitle_dest, local_srt_path)
  end

  def copy_to_(path, _to_path, asset) when path in ["None_Required", "order_captions"],
    do: {:ok, path}

  def copy_to_(from_path, to_path, asset) do
    from_path = String.replace(from_path, " ", "+")
    to_path = String.replace(to_path, " ", "+")
    local_download_path = download_s3(from_path, Utils.get_extname(from_path), asset)

    :ok = upload_stream(local_download_path, to_path)

    {:ok, local_download_path}
  end

  @doc """
  Transcode Asset - Transcode Asset into .flv for Streaming
  """
  def transcode_asset(local_raw_path, asset) do
    local_transcode_path = create_media_file(".flv", asset)

    :ok = run_shell("ffmpeg \
      -y \
      -i \"#{local_raw_path}\" \
      -threads 0 \
      -vcodec libx264 \
      -profile:v main \
      -preset:v medium \
      -r 30 \
      -g 60 \
      -keyint_min 60 \
      -sc_threshold 0 \
      -b:v 4000k \
      -minrate 4000k \
      -maxrate 4000k \
      -bufsize 4000k \
      -filter:v scale=\"iw*min(1920/iw\\,1080/ih):ih*min(1920/iw\\,1080/ih), pad=1920:1080:(1920-iw*min(1920/iw\\,1080/ih))/2:(1080-ih*min(1920/iw\\,1080/ih))/2\" \
      -pix_fmt yuv420p \
      -sws_flags lanczos+accurate_rnd \
      -strict -2 \
      -acodec aac \
      -b:a 96k \
      -ar 48000 \
      -ac 2 \
      -f flv \
      #{local_transcode_path}")

    {:ok, local_transcode_path}
  end

  @doc """
  Get Transcode Duration to prep for captions From Rev
  """
  def get_transcode_duration(local_transcode_path, asset) do
    {:ok, transcode_url} =
      ExAws.S3.presigned_url(@exaws_config, :get, @video_bucket, s3_path(:transcode, asset))

    raw_duration = FFprobe.get_duration(transcode_url)
    {:ok, asset} = Asset.update(asset, %{raw_duration: round(raw_duration / 1000)})

    if raw_duration > 0 do
      {:ok, asset}
    else
      {
        :error,
        "duration",
        "Asset duration not greater than 0: Asset #{asset.id}, duration: #{raw_duration}"
      }
    end
  end

  @doc """
  Check Subtitle Exist in S3, Need ordering, or Not required
  """
  def check_subtitles(%{source_subtitles: "None_Required"}), do: {:ok, :none_required}

  def check_subtitles(asset) do
    case s3_path_exists?(s3_path(:subtitles, asset)) do
      :ok -> {:ok, :validate_captions}
      :error -> {:ok, :order_captions}
    end
  end

  @doc """
  Input Asset and Make Order for Subtitles to Rev
  """
  def order_captions(:none_required, _asset), do: {:ok, :none_required}
  def order_captions(:validate_captions, _asset), do: {:ok, :validate_captions}

  def order_captions(:order_captions, asset) do
    case RevCaptions.check_order_asset(asset) do
      {:ok, %HTTPoison.Response{} = input_response} ->
        do_order_captions(input_response, asset)

      nil ->
        {:ok, :check_order_status}
    end
  end

  defp do_order_captions(%HTTPoison.Response{headers: headers}, asset) do
    Logger.info("Asset ID: #{asset.id}) Making order of asset to rev")

    location =
      headers
      |> Enum.into(%{})
      |> Map.get("Location")

    %{status_code: 201} = RevCaptions.order_captions(asset, location)
    {:ok, :check_order_status}
  end

  @doc """
  Check Subtitle Order Status From Rev
  """
  def check_order_status(order, asset)
  def check_order_status(:none_required, _asset), do: {:ok, :none_required}
  def check_order_status(:validate_captions, _asset), do: {:ok, :order_complete}

  def check_order_status(:check_order_status, asset) do
    case RevCaptions.get_status_of_order(asset) do
      "Complete" ->
        %{status_code: 200} = RevCaptions.get_captions(asset)
        {:ok, :order_complete}

      nil ->
        Logger.info(
          "(Asset ID: #{asset.id}) Could not get Status. Waiting on making order to Rev"
        )

        {:subtitle_order, :making_order}

      _ ->
        Logger.info("(Asset ID: #{asset.id}) Waiting on Captions from Rev")
        {:subtitle_order, :waiting}
    end
  end

  @doc """
  Validate Captions - Captions Valid if Caption file is greater than 10 bytes
  """
  def validate_captions(:none_required, _asset), do: {:ok, :none_required}

  def validate_captions(:order_complete, asset) do
    s3_subtitles_path = s3_path(:subtitles, asset)
    subtitle_ext = asset_extname(:subtitles, asset)

    with true <- subtitle_ext in [".srt", ".scc"],
         {:ok, _size} <- atleast_min_size(s3_subtitles_path, @min_byte_size, :s3),
         local_subtitle_path <- download_s3(s3_subtitles_path, subtitle_ext, asset),
         {:ok, _size} <- atleast_min_size(local_subtitle_path, @min_byte_size, :local) do
      {:ok, local_subtitle_path}
    else
      _result -> {:error, "captions", "(Asset ID: #{asset.id}) Invalid Captions"}
    end
  end

  @doc """
  Apply Captions - Applies Captions to .flv video. Supports only .scc and .srt
  """
  def apply_captions(:none_required, local_transcode_path, _asset),
    do: {:ok, local_transcode_path}

  def apply_captions(local_subtitle_path, local_transcode_path, asset) do
    local_captioned_path = create_media_file(".flv", asset)

    captioner =
      case Utils.get_extname(local_subtitle_path) do
        ".srt" -> "flv+srt"
        ".scc" -> "flv+scc"
      end

    :ok =
      run_shell(
        "#{captioner} #{local_transcode_path} #{local_subtitle_path} #{local_captioned_path}"
      )

    {:ok, local_captioned_path}
  end

  @doc """
  ts Convert - Converts flv to mpegts format
  """
  def ts_convert(local_flv_path, asset) do
    local_ts_path = create_media_file(".ts", asset)
    :ok = run_shell("ffmpeg -y -i #{local_flv_path} -c copy #{local_ts_path}")
    {:ok, local_ts_path}
  end

  @doc """
  Trim Asset - Trim segments out of asset
  """
  def trim_asset(local_ts_path, %{trimmings: ""} = asset), do: {:ok, local_ts_path}

  def trim_asset(local_ts_path, asset) do
    with local_trim_path <- create_media_file(".ts", asset),
         true <- Transcode.Trimmer.validate(asset.trimmings, asset),
         :ok <- Transcode.Trimmer.trim(local_ts_path, asset.trimmings, local_trim_path) do
      {:ok, local_trim_path}
    else
      false ->
        {:error, "trimmings", "invalid trimmings, a timestamp is greater than asset duration"}
    end
  end

  @doc """
  Validating Asset - FFprobe S3 URL for duration. If final duration matches transcode duration, then valid.
  """
  def final_validation(%{status: "trim"} = asset, _dependency) do
    with {:ok, final_url} <-
           ExAws.S3.presigned_url(@exaws_config, :get, @video_bucket, asset.s3_path),
         duration when is_integer(duration) <- FFprobe.get_duration(final_url),
         true <- duration > 0 do
      {:ok, asset} = Asset.update(asset, %{duration: duration})
    else
      false ->
        {
          :validation_error,
          "Asset ID #{asset.id}: Asset duration of #{asset.duration} is not greater than 0"
        }

      nil ->
        {:validation_error, "Asset ID #{asset.id}: S3 Path does not exist - #{asset.s3_path}"}

      err ->
        {:validation_error, err}
    end
  end

  @doc """
  Complete Asset - Complete
  """
  def complete(%{duration: duration} = asset) when is_integer(duration) do
    {:ok, asset} = update_asset("complete", asset, duration)
  end

  ### Helpers ###
  def s3_path_exists?(s3_path) do
    {status, _resp} =
      ExAws.S3.head_object(@video_bucket, s3_path)
      |> ExAws.request(@exaws_config)

    status
  end

  def update_asset(state, asset, _dependency) do
    Logger.info("(Asset ID: #{asset.id}) #{state}")
    {:ok, asset} = Asset.update(asset, %{status: state})
  end

  def create_media_file(extname, asset) do
    {:ok, local_path} = Utils.create_tmp_file(extname)
    local_path
  end

  def run_shell(cmd) do
    case Porcelain.shell(cmd) do
      %{status: 0} -> :ok
      error -> {:shell_error, error}
    end
  end

  def upload_file(s3_path, local_path) do
    with {:ok, size} <- atleast_min_size(local_path, @min_byte_size, :local),
         extname when extname != :no_extname <- Utils.get_extname(local_path),
         :ok <- do_upload(local_path, s3_path, size),
         {:ok, _size} <- atleast_min_size(s3_path, @min_byte_size, :s3) do
      {:ok, local_path}
    else
      result -> {:error, "upload", result}
    end
  end

  defp do_upload(local_path, destination, size) when size < @five_gb,
    do: upload_curl(local_path, destination)

  defp do_upload(local_path, destination, size), do: upload_stream(local_path, destination)

  defp upload_curl(local_path, s3_path) do
    {:ok, upload_url} = ExAws.S3.presigned_url(@exaws_config, :put, @video_bucket, s3_path)

    content_type_header =
      local_path
      |> Utils.get_extname()
      |> Utils.content_type()

    run_shell(
      "curl -X PUT -H \"Content-Type: #{content_type_header}\" --upload-file \"#{local_path}\" \"#{
        upload_url
      }\""
    )
  end

  defp upload_stream(local_path, s3_path) do
    file_size_bytes = content_length(local_path, :local)
    upload_chunk_size = round(file_size_bytes / 9_900)
    stream_opts = [chunk_size: max(upload_chunk_size, @five_mb)]
    upload_opts = [max_concurrency: 8, timeout: 60_000]

    {:ok, %{status_code: 200}} =
      local_path
      |> ExAws.S3.Upload.stream_file(stream_opts)
      |> ExAws.S3.upload(@video_bucket, s3_path, upload_opts)
      |> ExAws.request(@exaws_config)

    :ok
  end

  def download_s3(s3_path, extname, asset) do
    local_path = create_media_file(extname, asset)

    {:ok, :done} =
      ExAws.S3.download_file(@video_bucket, s3_path, local_path)
      |> ExAws.request(@exaws_config)

    local_path
  end

  defp atleast_min_size(path, min_size, location \\ :local) do
    size = content_length(path, location)

    case size >= min_size do
      true -> {:ok, size}
      _ -> {:error, "#{path}: size of #{size} bytes is less than minimum #{min_size} bytes"}
    end
  end

  def content_length(path, location) do
    case get_content_length(path, location) do
      size when is_integer(size) -> size
      error -> raise error
    end
  end

  def get_content_length(local_path, :local) do
    retry_while with: lin_backoff(500, 2) |> expiry(10_000) do
      try do
        {:halt, File.stat!(local_path).size}
      rescue
        error -> {:cont, error}
      end
    end
  end

  def get_content_length(s3_path, :s3) do
    retry_while with: lin_backoff(1000, 2) |> expiry(180_000) do
      try do
        %{status_code: 200, headers: caption_headers} =
          ExAws.S3.head_object(@video_bucket, s3_path)
          |> ExAws.request!(@exaws_config)

        {size, _rem} =
          caption_headers
          |> Enum.into(%{})
          |> Map.get("Content-Length")
          |> Integer.parse()

        {:halt, size}
      rescue
        error -> {:cont, error}
      end
    end
  end

  def s3_path(state, asset) do
    dest = %{
      raw: "/raw/",
      transcode: "/transcode/",
      subtitles: "/subtitles/",
      captioned: "/captioned/",
      ts: "/ts/",
      trim: "/trim/",
      final: "/final/"
    }

    String.replace(asset.s3_path, "/final/", dest[state])
    |> String.replace(Utils.get_extname(asset.s3_path), asset_extname(state, asset))
    |> String.replace(" ", "+")
  end

  def asset_extname(state, _asset) when state in [:captioned, :transcode], do: ".flv"
  def asset_extname(state, _asset) when state in [:ts, :trim, :final], do: ".ts"
  def asset_extname(:raw, asset), do: Utils.get_extname(asset.source_movie)

  def asset_extname(:subtitles, %{source_subtitles: source_subtitles})
      when source_subtitles in ["None_Required", "order_captions"],
      do: ".srt"

  def asset_extname(:subtitles, asset), do: Utils.get_extname(asset.source_subtitles)

  def log_error({:error, state, message}, asset) do
    Logger.error(message)
    Asset.update(asset, %{status: "error-#{state}"})
  end
end
