defmodule Transcode.Trimmer do
  require Utils
  require ConcatList
  # validate that time does not exceed total duration. (will cause ffmpeg to lockup)
  def validate(nil, %MissionControlEx.Web.Asset{}), do: true

  def validate(trimmings, asset) do
    trimmings
    |> format_trimmings
    |> Enum.map(&convert_timestamp_to_seconds/1)
    |> Enum.all?(fn
         "end" -> true
         time -> time < asset.raw_duration && time >= 0
       end)
  end

  def trim(local_video_path, trimmings, local_dest_path) do
    concat_file_list =
      trimmings
      |> format_trimmings
      |> create_trims(local_video_path)

    Transcode.run_shell("ffmpeg -y \
    -f concat -safe 0 \
    -i #{concat_file_list} \
    -c copy \
    #{local_dest_path}")
  end

  def format_trimmings(trimmings) do
    trimmings
    |> String.trim()
    |> String.split("\n")
    |> Enum.map(&String.trim/1)
    |> Enum.map(&String.split(&1, ","))
    |> List.flatten()
    |> Enum.map(&String.trim/1)
  end

  def create_trims(trimmings, video_path) do
    (["00:00:00" | trimmings] ++ ["end"])
    |> Enum.chunk(2)
    |> Enum.map(&trim_from_to_cmd(&1, video_path))
    |> Enum.reject(fn path -> path == nil end)
    |> Enum.map(&create_trim/1)
    |> create_concat_list
  end

  def trim_from_to_cmd([same_time, same_time], _video_path), do: nil

  def trim_from_to_cmd([start_time, "end"], video_path),
    do: "ffmpeg -y -i #{video_path} -ss #{start_time} -c copy -copyts -avoid_negative_ts 1"

  def trim_from_to_cmd([start_time, end_time], video_path),
    do:
      "ffmpeg -y -i #{video_path} -ss #{start_time} -to #{end_time} -c copy -copyts -avoid_negative_ts 1"

  def create_trim(cmd) do
    {:ok, local_output} = Briefly.create(extname: ".ts")
    :ok = Transcode.run_shell("#{cmd} #{local_output}")
    local_output
  end

  def create_concat_list(file_paths) do
    {:ok, file_list_path} = Briefly.create(extname: ".txt")
    File.write!(file_list_path, ConcatList.file_list(file_paths))
    file_list_path
  end

  def convert_timestamp_to_seconds("end"), do: "end"

  def convert_timestamp_to_seconds(timestamp) do
    [hh, mm, ss_ms] =
      timestamp
      |> String.split(":")
      |> Enum.map(fn time -> Float.parse(time) end)
      |> Enum.map(fn {float, _rem} -> float end)

    hh * 3600 + mm * 60 + ss_ms
  end
end
