defmodule MissionControlEx.Web.Asset do
  use MissionControlEx.Web, :model
  alias Ecto.Multi
  require Utils
  require Logger
  @bucket "twitch-creative-video-repository"

  defimpl IndifferentAccess, for: MissionControlEx.Web.Asset do
    def get(data, key), do: Map.get(data, key)
  end

  @derive {
            Poison.Encoder,
            only: [
              :id,
              :source_movie,
              :source_subtitles,
              :status,
              :s3_path,
              :s3_transcode_path,
              :duration,
              :raw_duration,
              :commercial_breaks_ms,
              :type,
              :metadata
            ]
          }

  schema "assets" do
    field(:source_movie, :string, default: nil)
    field(:source_subtitles, :string, default: "None_Required")
    field(:status, :string, default: "waiting")
    field(:s3_path, :string)
    field(:duration, :integer)
    field(:raw_duration, :integer)
    field(:commercial_breaks_ms, {:array, :integer}, default: [])
    field(:trimmings, :string, default: "")
    field(:metadata, :map, default: %{})
    field(:type, :string, virtual: true)

    many_to_many(
      :manifests,
      MissionControlEx.Web.Manifest,
      join_through: "manifests_assets",
      on_replace: :delete
    )

    timestamps()
  end

  @asset_key [:source_movie, :trimmings]
  @required [:source_movie, :status, :s3_path]
  @optional [
    :source_subtitles,
    :duration,
    :type,
    :raw_duration,
    :updated_at,
    :commercial_breaks_ms,
    :trimmings,
    :metadata
  ]

  def get(id) do
    Repo.get(__MODULE__, id)
  end

  # def get_next_asset(history, %{"asset_type" => type, "ordering" => %{"blocks_of" => %{"unit" => blocks_of_unit, "quantity" => blocks_of_quantity, "times_to_repeat" => times_to_repeat}}}) do
  def get_next_asset(%{events: history} = twitch_stream, schema_chunk) do
    asset_play_count =
      history
      |> Enum.take_while(fn event -> !is_reset_event(event, schema_chunk) end)
      |> Enum.filter(fn event -> play_asset_matching_schema_chunk?(event, schema_chunk) end)
      |> Enum.count()

    try do
      schedule_assets(schema_chunk, asset_play_count)
    rescue
      _ in RuntimeError -> nil
    end
  end

  def play_asset_matching_schema_chunk?(%{"start_time" => _}, _), do: false

  def play_asset_matching_schema_chunk?(
        %{type: "play_asset", ref: ref, data: %{"asset" => %{}}},
        %{"ref" => ref}
      )
      when ref != nil,
      do: true

  def play_asset_matching_schema_chunk?(_, _), do: false

  def schedule_assets(%{"id" => id} = schema_chunk, asset_play_count) do
    asset = get(id)
    number_of_indices = length(asset.commercial_breaks_ms)

    Enum.map(0..number_of_indices, fn index -> {asset, index, number_of_indices} end)
    |> apply_shuffling_and_cycling(schema_chunk)
    |> Stream.drop(asset_play_count)
    |> Enum.take(1)
    |> List.first()
  end

  def schedule_assets(schema_chunk, asset_play_count) do
    {:ok, result} =
      Repo.transaction(fn ->
        get_assets_by_schema_chunk(schema_chunk)
        |> get_scheduled_asset(schema_chunk, asset_play_count)
      end)

    result
  end

  defp get_scheduled_asset(
         assets,
         %{
           "ordering" => %{
             "blocks_of" => %{"unit" => "assets", "quantity" => blocks_of_quantity},
             "times_to_repeat" => times_to_repeat
           }
         } = schema_chunk,
         asset_play_count
       )
       when length(assets) > 0 do
    assets
    |> apply_shuffling_and_cycling(schema_chunk)
    |> Stream.chunk(blocks_of_quantity, blocks_of_quantity, [])
    |> Stream.flat_map(fn chunk ->
         Stream.flat_map(1..times_to_repeat, fn _ ->
           Stream.flat_map(chunk, fn %{commercial_breaks_ms: commercial_breaks_ms} = asset ->
             number_of_indices = length(commercial_breaks_ms)
             Stream.map(0..number_of_indices, fn index -> {asset, index, number_of_indices} end)
           end)
         end)
       end)
    |> Stream.drop(asset_play_count)
    |> Enum.take(1)
    |> List.first()
  end

  defp get_scheduled_asset(_, _, _), do: raise("No assets found with given filters")

  def get_assets_by_schema_chunk(
        %{"order_by" => "__shuffle__", "filter_groups" => [filter_group]} = schema_chunk
      ) do
    q =
      from(a in __MODULE__, where: ^build_dynamic_filter(filter_group))
      |> Repo.all()
  end

  def get_assets_by_schema_chunk(
        %{"order_by" => meta_field, "filter_groups" => [filter_group]} = schema_chunk
      ) do
    q =
      from(
        a in __MODULE__,
        where: ^build_dynamic_filter(filter_group),
        order_by: fragment("metadata ->> ?", ^meta_field)
      )
      |> Repo.all()
      |> Enum.sort(&(&1.metadata[meta_field] <= &2.metadata[meta_field]))
  end

  def build_dynamic_filter(%{"grouping" => "and", "filters" => filters}) do
    dynamic(
      [],
      ^Enum.reduce(filters, true, fn filter, accum ->
        dynamic([], ^accum and ^build_dynamic_filter(filter))
      end)
    )
  end

  def build_dynamic_filter(%{"grouping" => "or", "filters" => filters}) do
    dynamic(
      [],
      ^Enum.reduce(filters, false, fn filter, accum ->
        dynamic([], ^accum or ^build_dynamic_filter(filter))
      end)
    )
  end

  def build_dynamic_filter(%{"type" => "filter", "key" => key, "val" => val} = filter),
    do: dynamic([], fragment("metadata ->> ? = ?", ^to_string(key), ^to_string(val)))

  def apply_filter(%{"key" => key, "val" => val}, %{metadata: metadata}), do: metadata[key] == val

  def apply_filter(%{"type" => "filter_group", "grouping" => "or", "filters" => filters}, asset),
    do: Enum.any?(filters, fn filter -> apply_filter(filter, asset) end)

  def apply_filter(%{"type" => "filter_group", "grouping" => "and", "filters" => filters}, asset),
    do: Enum.all?(filters, fn filter -> apply_filter(filter, asset) end)

  def apply_shuffling_and_cycling(assets, %{"on_empty" => on_empty, "order" => order}) do
    Stream.resource(
      fn -> 0 end,
      fn run_index ->
        case {order, on_empty, run_index} do
          {_, on_empty, 1} when on_empty != "reset" ->
            {:halt, run_index}

          {"shuffle", _, _} ->
            set_seed(run_index)
            {Enum.shuffle(assets), run_index + 1}

          {_, _, _} ->
            {assets, run_index + 1}
        end
      end,
      fn _ -> nil end
    )
  end

  def apply_shuffling_and_cycling(assets, _), do: assets

  defp set_seed(run_index), do: :rand.seed(:exs1024, {42, 42, run_index})

  def filter_out_assets_that_can_still_play(asset_ids, 0, "assets", _), do: asset_ids

  def filter_out_assets_that_can_still_play(
        asset_ids,
        times_to_repeat,
        "assets",
        blocks_of_quantity
      ) do
    exhausted_asset_count =
      trunc(length(asset_ids) / (times_to_repeat * blocks_of_quantity)) *
        (times_to_repeat * blocks_of_quantity)

    {exhausted_assets, assets_in_current_iteration} = Enum.split(asset_ids, exhausted_asset_count)

    max_number_of_plays =
      assets_in_current_iteration
      |> Enum.group_by(& &1)
      |> Enum.map(fn {thing, plays} -> length(plays) end)
      |> Enum.concat([0])
      |> Enum.max()

    finished_loop_count =
      trunc(length(assets_in_current_iteration) / blocks_of_quantity) * blocks_of_quantity

    {finished_assets, assets_in_current_loop} =
      Enum.split(assets_in_current_iteration, finished_loop_count)

    exhausted_assets ++ assets_in_current_loop
  end

  def is_reset_event(%{type: "reset_stream"}, _), do: true

  def is_reset_event(%{type: "reset_asset", data: %{"asset_type" => asset_type}}, %{
        "asset_type" => asset_type
      }),
      do: true

  def is_reset_event(_, _), do: false

  def same?(a, b), do: a.source_movie == b.source_movie and a.trimmings == b.trimmings

  def all() do
    Repo.all(__MODULE__)
  end

  def all_order() do
    Repo.all(from(a in __MODULE__, order_by: a.id))
  end

  def find_by_asset_ids(asset_ids) do
    Repo.all(from(a in __MODULE__, where: a.id in ^asset_ids))
  end

  def find_by_asset_id(asset_id) do
    Repo.one(from(a in __MODULE__, where: a.id == ^asset_id))
  end

  def get_asset_duration(nil), do: nil
  def get_asset_duration(asset), do: asset.duration

  def update(asset, map) do
    Repo.update(changeset(asset, map))
  end

  def create_new_asset(raw_s3_path, overrides \\ %{}) do
    create_new_asset(asset_exists?(raw_s3_path), raw_s3_path, overrides)
  end

  def create_new_asset(true, raw_s3_path, _overrides),
    do: Logger.error("Asset already exists in db: #{raw_s3_path}")

  def create_new_asset(false, raw_s3_path, overrides),
    do: new_asset_from_s3_path(raw_s3_path, overrides)

  def asset_exists?(raw_s3_path) do
    db_result =
      Repo.all(from(a in MissionControlEx.Web.Asset, where: a.source_movie == ^raw_s3_path))

    exists?(db_result)
  end

  def exists?([]), do: false
  def exists?(db_results) when length(db_results) > 0, do: true
  def exists?(_), do: raise("Error checking if an asset exists")

  def new_asset_from_s3_path(s3_path, overrides \\ %{}) do
    default = %MissionControlEx.Web.Asset{
      s3_path: s3_path,
      duration: 0,
      status: "complete"
    }

    case Repo.insert(changeset(default, overrides)) do
      {:ok, asset} ->
        asset

      {:error, _error} ->
        asset =
          Repo.get_by(MissionControlEx.Web.Asset, %{s3_path: overrides["s3_path"]})
          |> Repo.preload(:manifests)

        asset
        |> Ecto.Changeset.change()
        |> put_assoc(:manifests, asset.manifests ++ overrides["manifests"])
        |> Repo.update!()
    end
  end

  def touch(asset) do
    Repo.update!(changeset(asset, %{updated_at: NaiveDateTime.utc_now()}))
  end

  def get_chunk_start_and_duration(%{duration: duration} = asset, "whole_asset"),
    do: {0, duration}

  def get_chunk_start_and_duration(%{"duration" => duration} = asset, "whole_asset"),
    do: {0, duration}

  def get_chunk_start_and_duration(%{} = asset, index) when index > 0 do
    start = Enum.at(IndifferentAccess.get(asset, :commercial_breaks_ms), index - 1) || 0

    {
      start,
      (Enum.at(IndifferentAccess.get(asset, :commercial_breaks_ms), index) ||
         IndifferentAccess.get(asset, :duration)) - start
    }
  end

  def get_chunk_start_and_duration(%{} = asset, 0) do
    {
      0,
      Enum.at(IndifferentAccess.get(asset, :commercial_breaks_ms), 0) ||
        IndifferentAccess.get(asset, :duration)
    }
  end

  def get_chunk_start_and_duration(asset_id, index),
    do: get_chunk_start_and_duration(get(asset_id), index)

  def get_signed_url(%{"s3_path" => s3_path}), do: sign_url(s3_path)
  def get_signed_url(%{s3_path: s3_path}), do: sign_url(s3_path)

  def sign_url(s3_path) do
    {:ok, url} =
      ExAws.S3.presigned_url(
        ExAws.Config.new(:s3, %{
          access_key_id: "AKIAJ4RHGC5TBNDDSR3A",
          secret_access_key: "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"
        }),
        :get,
        @bucket,
        s3_path
      )

    url
  end

  def get_asset_from_rows(rows) do
    q =
      Enum.reduce(rows, __MODULE__, fn row, query ->
        from(q in query, or_where: ^Enum.map(@asset_key, fn key -> {key, Map.get(row, key)} end))
      end)

    Repo.all(q)
  end

  def get_assets_by_metafield(field) do
    from(
      a in __MODULE__,
      where: fragment("metadata \\? ?", ^field),
      order_by: fragment("metadata ->> ?", ^field)
    )
    |> Repo.all()
    |> Enum.sort_by(fn asset -> asset.metadata[field] end)
  end

  def update_metafields(asset_field_values) do
    asset_field_values
    |> Enum.reduce(Multi.new(), fn {asset, %{} = changes}, multi ->
         new_metadata = Enum.reduce(changes, asset.metadata, &apply_metadata_changes/2)
         Multi.update(multi, Utils.random_string(64), changeset(asset, %{metadata: new_metadata}))
       end)
    |> Repo.transaction()
  end

  defp apply_metadata_changes({field, "__delete__"}, metadata), do: Map.delete(metadata, field)
  defp apply_metadata_changes({field, value}, nil), do: %{field => value}

  defp apply_metadata_changes({field, value}, metadata) when is_binary(value) do
    new_value =
      case seq_end?(field) do
        true ->
          {int_seq, _} = Integer.parse(value)
          int_seq

        _ ->
          value
      end

    Map.put(metadata, field, new_value)
  end

  defp apply_metadata_changes({field, value}, metadata), do: Map.put(metadata, field, value)

  def update_metafields(asset, field, "__delete__"),
    do: changeset(asset, %{metadata: Map.delete(asset.metadata, field)})

  def update_metafields(asset, field, value) when is_binary(value) do
    new_value =
      case seq_end?(field) do
        true ->
          {int_seq, _} = Integer.parse(value)
          int_seq

        _ ->
          value
      end

    changeset(asset, %{metadata: Map.put(asset.metadata, field, new_value)})
  end

  def update_metafields(asset, field, value),
    do: changeset(asset, %{metadata: Map.put(asset.metadata, field, value)})

  defp seq_end?(metafield), do: Regex.match?(~r/(_seq)$/, metafield)

  @doc """
  Builds a changeset based on the `struct` and `params`.
  """
  def changeset(struct, params) when struct == params, do: Ecto.Changeset.change(struct)

  def changeset(struct, params \\ %{}) do
    struct
    |> cast(params, @optional ++ @required)
    |> validate_required(@required)
    |> unique_constraint(
         :unique_source_movie_and_trimmings,
         name: :assets_source_movie_trimmings_index
       )
  end
end
