defmodule MissionControlEx.Web.Channel do
  use MissionControlEx.Web, :model

  schema "channels" do
    field(:current_twitch_stream_id, :integer)
    field(:login, :string)
    field(:oauth_token, :string)
    field(:twitch_profile_id, :integer)
    field(:chat_bot_enabled, :boolean, default: false)
    field(:chat_data, :map, default: %{}, virtual: true)
    has_one(:schedule_manager, MissionControlEx.Web.ScheduleManager, foreign_key: :channel_id)
    many_to_many(:twitch_streams, MissionControlEx.Web.StreamSchedule, join_through: "airings")

    has_many(:channel_commands, MissionControlEx.Web.ChannelCommand, foreign_key: :channel_id)
    has_many(:chat_commands, through: [:channel_commands, :chat_command])
    timestamps()
  end

  @client_id Application.get_env(:ueberauth, Ueberauth.Strategy.TwitchTv.OAuth)[:client_id]

  @required [:login]
  @optional [:current_twitch_stream_id, :oauth_token, :twitch_profile_id, :chat_bot_enabled]

  def get_channel_by_login(channel_login) do
    channel = Repo.get_by(__MODULE__, %{login: channel_login})
    load(channel.id)
  end

  def create_from_oauth(ueberauth_auth) do
    channel =
      MissionControlEx.Web.TwitchApi.get!(
        "kraken/channel",
        "Client-ID": @client_id,
        Authorization: "OAuth #{ueberauth_auth.credentials.token}"
      )
      |> Map.get(:body)

    changes = %{
      login: channel["name"],
      oauth_token: ueberauth_auth.credentials.token
    }

    case Repo.get_by(__MODULE__, twitch_profile_id: channel["_id"]) do
      nil -> %__MODULE__{twitch_profile_id: channel["_id"]}
      channel -> channel
    end
    |> changeset(changes)
    |> Repo.insert_or_update()
  end

  def get_stream_key(%__MODULE__{login: login, oauth_token: oauth_token} = channel) do
    MissionControlEx.Web.TwitchApi.get!(
      "kraken/channel",
      "Client-ID": @client_id,
      Authorization: "OAuth #{oauth_token}"
    )
    |> Map.get(:body)
    |> Map.get("stream_key")
    |> case do
         nil -> raise "stream_key is nil for channel #{login}; likely invalid oauth token"
         stream_key -> stream_key
       end
  end

  def get_channel(nil), do: nil
  def get_channel(channel_id), do: load(channel_id)

  def get_current_twitch_stream(
        %MissionControlEx.Web.Channel{current_twitch_stream_id: twitch_stream_id} = channel
      ) do
    MissionControlEx.Web.StreamSchedule.get_stream(twitch_stream_id)
  end

  def get_current_twitch_stream(channel_id) do
    chan = load(channel_id)
    Repo.get_by(MissionControlEx.Web.StreamSchedule, id: chan.current_twitch_stream_id)
  end

  def all(), do: Repo.all(__MODULE__)

  @doc """
  Returns the played asset history for a given channel
  """
  def get_history(%{id: channel_id}), do: get_history(channel_id)

  def get_history(id) do
    Repo.all(
      from(
        a in MissionControlEx.Web.Asset,
        join: c in assoc(a, :channels),
        join: c_a in assoc(a, :events),
        where: c.id == ^id,
        order_by: c_a.inserted_at,
        select: a
      )
    )
  end

  def set_current_twitch_stream(channel_id, twitch_stream_id) do
    with channel <- load(channel_id),
         changes <- Map.from_struct(%{channel | current_twitch_stream_id: twitch_stream_id}) do
      changeset(channel, changes)
      |> Repo.update!()
    end
  end

  def get_current_airing(channel_id) do
    # TODO: Make this return the "current" airing for channels with multiple airings
    Repo.one(
      from(
        a in MissionControlEx.Web.ScheduleManager,
        join: c in assoc(a, :channel),
        where: c.id == ^channel_id,
        limit: 1
      )
    )
  end

  @doc """
  Builds a changeset based on the `struct` and `params`.
  """
  def changeset(struct, params \\ %{}) do
    struct
    |> cast(params, @required ++ @optional)
    |> validate_required(@required)
    |> unique_constraint(:login)
  end

  def load(channel_id) do
    Repo.one(
      from(
        c in __MODULE__,
        where: c.id == ^channel_id,
        preload: [:chat_commands]
      )
    )
  end

  def load_with_chat_data(%{id: id}), do: load_with_chat_data(id)

  def load_with_chat_data(channel_id) do
    channel =
      Repo.one(
        from(
          c in __MODULE__,
          where: c.id == ^channel_id,
          preload: [:chat_commands, :schedule_manager]
        )
      )
      |> Map.put(:chat_data, %{})
      |> load_schedule_manager
      |> load_current_data
      |> load_next_data
  end

  defp load_schedule_manager(%{schedule_manager: schedule_manager} = channel),
    do: %{
      channel
      | schedule_manager: MissionControlEx.Web.ScheduleManager.load(schedule_manager.id)
    }

  defp reverse_events(%{schedule_manager: %{events: events} = schedule_manager} = channel),
    do: %{channel | schedule_manager: %{schedule_manager | events: Enum.reverse(events)}}

  def load_current_data(
        %{chat_data: chat_data, schedule_manager: %{events: events} = schedule_manager} = chan
      ) do
    try do
      event = MissionControlEx.Web.Event.get_last_event_with_chat_data(schedule_manager)
      chat_data = Map.put(chat_data, :current_play_event, event)
      put_in_channel_chat_data(%{chan | chat_data: chat_data}, "current", event.data["chat_data"])
    rescue
      _ -> chan
    end
  end

  def load_current_data(chan), do: chan

  defp load_next_data(
         %{
           chat_data: %{current_play_event: current_event} = chat_data,
           schedule_manager: schedule_manager
         } = chan
       ) do
    try do
      next_event =
        MissionControlEx.Web.Event.get_next_unique_event(schedule_manager, current_event)

      put_in_channel_chat_data(chan, "next", next_event.data["chat_data"])
    rescue
      _ -> chan
    end
  end

  defp load_next_data(chan), do: chan

  defp put_in_channel_chat_data(%{chat_data: chat_data} = chan, key, metadata),
    do: %{chan | chat_data: Map.put(chat_data, key, metadata)}
end
