defmodule MissionControlEx.Web.Event do
  use MissionControlEx.Web, :model
  alias MissionControlEx.Web.{StreamSchedule, ScheduleManager, EventChunk}
  @twitch_stream_delay Application.get_env(:mission_control_ex, :twitch_delay)
  {:ok, endpoint} = Application.fetch_env(:mission_control_ex, :event_api_url)
  @api_endpoint endpoint
  {:ok, video_bytes} = File.read("videos/backup_asset.ts")
  @backup_bytes video_bytes
  @derive {
            Poison.Encoder,
            only: [
              :id,
              :data,
              :ref,
              :type,
              :inserted_at,
              :projected_inserted_at,
              :twitch_stream_id,
              :chunk_id,
              :chunk_index
            ]
          }

  {:ok, video_bytes} = File.read("videos/backup_asset.ts")
  @backup_bytes video_bytes

  schema "events" do
    belongs_to(
      :twitch_stream,
      MissionControlEx.Web.StreamSchedule,
      foreign_key: :twitch_stream_id
    )

    field(:data, :map, default: %{})
    field(:ref, :string)
    field(:type, :string)
    field(:projected_inserted_at, :naive_datetime, virtual: true)
    field(:chunk_id, :string)
    field(:chunk_index, :integer)
    timestamps()
  end

  def process(%__MODULE__{
        type: "play_commercial",
        data: %{"duration" => duration},
        twitch_stream_id: ss_id
      }) do
    Task.start(fn ->
      :timer.sleep(@twitch_stream_delay)

      StreamSchedule.get_live_channels(ss_id)
      |> Enum.each(fn channel -> MissionControlEx.Web.AdInterface.post(channel, duration) end)
    end)
  end

  def process(event), do: :noop

  def persist(%__MODULE__{} = event), do: Repo.insert!(event)

  def persist(nil, event), do: Repo.insert!(event)

  def persist(%{id: id} = stream_schedule, event) do
    StreamSchedule.persist_changes(stream_schedule)
    Repo.insert!(%{event | twitch_stream_id: id})
  end

  def process_persist_update(event) when is_bitstring(event) do
    Poison.decode!(event, as: %__MODULE__{})
    |> process_persist_update
  end

  def process_persist_update(%__MODULE__{twitch_stream_id: ss_id} = event) do
    with _ <- process(event),
         %__MODULE__{} = event <- persist(event),
         %StreamSchedule{} <- StreamSchedule.update_with_event(event),
         do: event
  end

  def api_call_to_persist(%__MODULE__{} = event) do
    url = "#{@api_endpoint}/events/process"

    headers = [
      {"Content-Type", "application/json"}
    ]

    HTTPoison.post(url, Poison.encode!(%{event: event}), headers)
    event
  end

  def to_video_bytes(
        %__MODULE__{
          type: type,
          data: %{"start_time" => start_time, "duration" => duration, "asset" => asset}
        } = event
      )
      when type in ["play_commercial", "play_asset", "recovery_event"] do
    url = MissionControlEx.Web.Asset.get_signed_url(asset)
    [{:bytes, FFMpeger.get_video_bytes(url, start_time, duration)}]
  end

  def to_video_bytes(%__MODULE__{
        type: "play_remote_asset",
        data: %{"start_time" => start_time, "duration" => duration, "url" => url}
      }) do
    [{:bytes, FFMpeger.get_video_bytes(url, start_time, duration)}]
  end

  def to_video_bytes(%__MODULE__{
        type: "fallback_asset",
        data: %{"duration" => duration, "asset" => asset}
      }) do
    url = MissionControlEx.Web.Asset.get_signed_url(asset)
    [{:bytes, FFMpeger.get_video_bytes(url, 0, duration, ["-stream_loop", "1"])}]
  end

  def to_video_bytes(%__MODULE__{type: "stream_black_buffer"}) do
    [{:bytes, for(_ <- 1..30, do: @backup_bytes)}]
  end

  def to_video_bytes(%__MODULE__{
        type: "play_curated_asset",
        data: %{"url" => url}
      }) do
    [{:bytes, FFMpeger.get_video_bytes_from_web_page(url)}]
  end

  def to_video_bytes(%__MODULE__{
        type: "play_twitch_clip",
        data: %{"chat_data" => clip_data, "url" => url}
      }) do
    {:ok, overlay_path} = Briefly.create(extname: ".webm")
    task = Task.async(fn -> FFMpeger.stream_clip(url, clip_data, overlay_path) end)
    [{:async_bytes, task}, {:func, fn -> File.rm!(overlay_path) end}]
  end

  def to_video_bytes(%__MODULE__{type: "stream_start"}) do
    [{:bytes, for(_ <- 1..30, do: @backup_bytes)}]
  end

  def to_video_bytes(_), do: [{:bytes, []}]

  def make_event_from_json(%{"data" => data, "ref" => ref, "type" => type}, twitch_stream_id) do
    %__MODULE__{data: data, ref: ref, type: type, twitch_stream_id: twitch_stream_id}
  end

  def play_asset(stream_id, ref, asset, index, message_template) do
    asset = Poison.encode!(asset) |> Poison.decode!()

    {start_time, duration} =
      MissionControlEx.Web.Asset.get_chunk_start_and_duration(asset["id"], index)

    %__MODULE__{
      twitch_stream_id: stream_id,
      ref: ref,
      data: %{
        "asset" => asset,
        "index" => index,
        "start_time" => start_time,
        "duration" => duration,
        "message_template" => message_template,
        "chat_data" => asset["metadata"]
      },
      type: "play_asset"
    }
  end

  def play_commercial(stream_id, ref, asset, duration) do
    asset = Poison.encode!(asset) |> Poison.decode!()

    %__MODULE__{
      twitch_stream_id: stream_id,
      ref: ref,
      data: %{"asset" => asset, "start_time" => 0, "duration" => duration},
      type: "play_commercial"
    }
  end

  def play_curated_asset(stream_id, ref, url, data \\ %{}) do
    %__MODULE__{
      twitch_stream_id: stream_id,
      ref: ref,
      data: Map.put(data, "url", url),
      type: "play_curated_asset"
    }
  end

  def play_remote_asset(stream_id, ref, url, data \\ %{}) do
    %__MODULE__{
      twitch_stream_id: stream_id,
      ref: ref,
      data: Map.put(data, "url", url),
      type: "play_remote_asset"
    }
  end

  def play_twitch_clip(stream_id, ref, url, data \\ %{}) do
    %__MODULE__{twitch_stream_id: stream_id, ref: ref, data: data, type: "play_twitch_clip"}
  end

  def play_from_mrss(stream_id, ref, feed_id, feed_item, message_template) do
    feed_item = Poison.encode!(feed_item) |> Poison.decode!()
    duration = feed_item["duration_s"] * 1000

    play_remote_asset(stream_id, ref, feed_item["content_url"], %{
      "feed_id" => feed_id,
      "duration" => duration,
      "metadata" => feed_item,
      "message_template" => message_template
    })
  end

  def stream_start(stream_id) do
    %__MODULE__{twitch_stream_id: stream_id, type: "stream_start", data: %{"duration" => 31_000}}
  end

  def fallback_asset(stream_id, ref, asset, duration) do
    asset = Poison.encode!(asset) |> Poison.decode!()

    %__MODULE__{
      twitch_stream_id: stream_id,
      ref: ref,
      data: %{"asset" => asset, "start_time" => 0, "duration" => duration},
      type: "fallback_asset"
    }
  end

  def stream_black_buffer(stream_id) do
    %__MODULE__{
      twitch_stream_id: stream_id,
      type: "stream_black_buffer",
      data: %{"duration" => 31_000}
    }
  end

  def exiting_ref(stream_id, ref, exit_ref) do
    %__MODULE__{
      twitch_stream_id: stream_id,
      ref: ref,
      data: %{"exit_ref" => exit_ref},
      type: "exit_ref"
    }
  end

  def skip(stream_id, ref) do
    %__MODULE__{twitch_stream_id: stream_id, ref: ref, data: %{}, type: "skip"}
  end

  def suspend(stream_id, ref, duration) do
    %__MODULE__{
      twitch_stream_id: stream_id,
      ref: ref,
      data: %{"duration" => duration},
      type: "suspend"
    }
  end

  def launch_stream(stream_id, ref) do
    %__MODULE__{
      twitch_stream_id: stream_id,
      ref: ref,
      data: %{"duration" => 0},
      type: "launch_stream"
    }
  end

  def schedule_done(stream_id) do
    %__MODULE__{twitch_stream_id: stream_id, ref: "___DONE___", data: %{}, type: "schedule_done"}
  end

  def end_stream(stream_id, chunk_id, recurring \\ false) do
    %__MODULE__{
      twitch_stream_id: stream_id,
      ref: "___DONE___",
      chunk_id: chunk_id,
      chunk_index: 0,
      data: %{"recurring" => recurring},
      type: "end_stream"
    }
  end

  def get_history(stream_id) do
    Repo.all(
      from(e in __MODULE__, where: e.twitch_stream_id == ^stream_id, order_by: [desc: :id])
    )
  end

  def get_last_event(%{twitch_stream_id: id, ref: ref, asset_type: type}) do
    Repo.one(
      from(
        e in __MODULE__,
        where: e.twitch_stream_id == ^id and e.ref == ^ref and e.asset_type == ^type,
        order_by: [desc: :id]
      )
    )
  end

  def get_next_unique_event(schedule_manager, event) do
    {_, event} =
      ScheduleManager.generate_event_stream(schedule_manager)
      |> Enum.flat_map_reduce(nil, fn %{event: next_event}, acc ->
           do_new_event(next_event, event)
         end)

    event
  end

  defp do_new_event(
         %{type: type, data: %{"asset" => %{"id" => next_id}}} = next_event,
         %{type: type, data: %{"asset" => %{"id" => current_id}}} = _current_event
       )
       when next_id != current_id,
       do: {:halt, next_event}

  defp do_new_event(
         %{type: "play_asset"} = next_event,
         %{type: "play_commercial"} = _current_event
       ),
       do: {:halt, next_event}

  defp do_new_event(
         %{type: "play_twitch_clip"} = next_event,
         %{type: "play_twitch_clip"} = _current_event
       ),
       do: {:halt, next_event}

  defp do_new_event(next_event, _), do: {[next_event], next_event}

  def get_last_event_of_type(%{events: events} = schedule_manager, type),
    do: do_get_last_of_type(Enum.reverse(events), type)

  defp do_get_last_of_type([%{type: type} = event | _], type), do: event
  defp do_get_last_of_type([_ | events], type), do: do_get_last_of_type(events, type)
  defp do_get_last_of_type([], _), do: {:error, :not_found}

  def get_last_event_with_chat_data(%{events: events} = schedule_manager) do
    Enum.reverse(events)
    |> Enum.find(fn
         %{data: %{"chat_data" => chat_data}} when chat_data != nil -> true
         _ -> false
       end)
  end

  def print_events(%{events: events}), do: do_print_events(events)

  def print_events(stream_id) do
    Repo.all(
      from(
        e in __MODULE__,
        where: e.twitch_stream_id == ^stream_id,
        order_by: [desc: :id]
      )
    )
    |> do_print_events
  end

  def do_print_events(%__MODULE__{} = event), do: do_print_events([event])

  def do_print_events(events) do
    IO.puts("")

    events
    |> Enum.map(fn event ->
         [
           event.type,
           event.ref,
           event.data["asset"]["s3_path"],
           event.data["index"],
           event.data["duration"],
           event.projected_inserted_at,
           event.chunk_index,
           event.inserted_at
         ]
       end)
    |> TableRex.quick_render!([
         "event_type",
         "ref",
         "s3_path",
         "index",
         "duration",
         "projected_inserted_at",
         "chunk_index",
         "inserted_at"
       ])
    |> IO.puts()

    events
  end

  def duration([h | t]), do: duration(h) + duration(t)
  def duration(%{data: %{"duration" => duration}}), do: duration

  def duration(%{type: "play_" <> _, data: %{"asset" => asset, "index" => index}}) do
    {_, duration} = MissionControlEx.Web.Asset.get_chunk_start_and_duration(asset, index)
    duration
  end

  def duration(_), do: 0

  def ended_at(%{inserted_at: inserted_at, data: %{"duration" => duration}}),
    do: Timex.add(inserted_at, Timex.Duration.from_milliseconds(duration))

  def ended_at(%{projected_inserted_at: projected_inserted_at, data: %{"duration" => duration}}),
    do: Timex.add(projected_inserted_at, Timex.Duration.from_milliseconds(duration))

  def ended_at(%{inserted_at: inserted_at}), do: inserted_at
  def ended_at(%{projected_inserted_at: projected_inserted_at}), do: projected_inserted_at

  @doc """
  Builds a changeset based on the `struct` and `params`.
  """
  def changeset(struct, params \\ %{})
  def changeset(struct, %__MODULE__{} = params), do: changeset(struct, Map.from_struct(params))

  def changeset(struct, params) do
    struct
    |> cast(params, [:twitch_stream_id, :data, :ref, :type])
    |> validate_required([:twitch_stream_id, :data, :ref, :type])
  end

  def cast(data) do
    data
    |> Poison.encode!()
    |> Poison.decode(as: %__MODULE__{})
  end
end
