defmodule MissionControlEx.Web.EventChunk do
  use MissionControlEx.Web, :model
  alias MissionControlEx.Web.{Event, EventChunk, ScheduleManager, StreamSchedule}

  @derive {
            Poison.Encoder,
            only: [
              :events,
              :data,
              :id,
              :stream_schedule_id
            ]
          }

  embedded_schema do
    field(:events, {:array, Event})
    field(:data, :map, default: %{})
    field(:stream_schedule_id, :integer)
  end

  def duration(%{events: events}), do: Event.duration(events)

  def link_schedule(chunk, %{id: id}) do
    %{
      chunk
      | events:
          Enum.map(chunk.events, fn event ->
            %{event | twitch_stream_id: id}
          end),
        stream_schedule_id: id
    }
  end

  def project_inserted_at(%{events: events} = chunk, start_time) do
    {events, _} =
      events
      |> Enum.map_reduce(start_time, fn event, acc ->
           updated_event = %{event | projected_inserted_at: acc}

           next_time =
             Timex.add(
               acc,
               Timex.Duration.from_milliseconds(Event.duration(event))
             )

           {updated_event, next_time}
         end)

    %{chunk | events: events}
  end

  def make_event_chunk(events, data \\ %{}) do
    chunk_id = Ecto.UUID.generate()
    make_event_chunk_with_id(events, chunk_id, data)
  end

  def make_event_chunk_with_id(events, chunk_id, data \\ %{}) do
    events =
      events
      |> Enum.with_index()
      |> Enum.map(fn {event, index} ->
           Map.merge(event, %{chunk_id: chunk_id, chunk_index: index})
         end)

    %__MODULE__{events: events, data: data, id: chunk_id}
  end

  def make_end_stream_chunk(%{id: id, repeat: repeat} = stream_schedule, data \\ %{}) do
    chunk_id = "END_STREAM_#{id}"
    recurring = repeat || length(stream_schedule.event_chunks) > 0
    events = [Event.end_stream(id, chunk_id, recurring)]

    %__MODULE__{events: events, data: data, id: chunk_id, stream_schedule_id: id}
  end

  def make_event_chunk_from_json(
        %{"id" => id, "data" => data, "events" => event_json},
        twitch_stream_id
      ) do
    events = Enum.map(event_json, &Event.make_event_from_json(&1, twitch_stream_id))
    %__MODULE__{events: events, data: data, id: id, stream_schedule_id: twitch_stream_id}
  end

  def chunk_stream_to_event_stream(stream) do
    Stream.flat_map(stream, fn %{events: events} -> events end)
  end

  def clip_to_chunk(clip_data, stream_schedule) do
    clip_data = %{clip_data | "duration" => clip_data["duration"] * 1000}
    chat_data = trim_clip_data(clip_data)

    event_data = %{
      "id" => clip_data["slug"],
      "duration" => clip_data["duration"],
      "chat_data" => chat_data,
      "url" => clip_data["url"]
    }

    new_event_chunk =
      [Event.play_twitch_clip(stream_schedule.id, "clip_mode", clip_data["url"], event_data)]
      |> EventChunk.make_event_chunk_with_id("CLIP-#{clip_data["slug"]}", event_data)
      |> EventChunk.link_schedule(stream_schedule)
  end

  def trim_clip_data(%{"broadcaster" => broadcaster, "curator" => curator, "vod" => vod} = data) do
    broadcaser = take_user_data(broadcaster)
    curator = take_user_data(curator)

    Map.take(data, ["url", "vod", "slug"])
    |> Map.put("broadcaster", broadcaster)
    |> Map.put("curator", curator)
  end

  def take_user_data(user_data), do: Map.take(user_data, ["channel_url", "name", "display_name"])

  def changeset(struct, params \\ %{})

  def changeset(struct, %__MODULE__{} = params), do: changeset(struct, Map.from_struct(params))

  def changeset(struct, params) do
    struct
    |> cast(params, [:events, :data])
    |> validate_required([])
  end
end
