defmodule MissionControlEx.Web.Feed do
  use MissionControlEx.Web, :model
  alias MissionControlEx.Web.Repo
  import SweetXml
  require Logger
  @done "___DONE___"
  @derive {Poison.Encoder, only: [:id, :url]}

  schema "feeds" do
    field(:url, :string)
    field(:next_guid, :string, default: nil)
    field(:xml, :string, virtual: true)
    field(:xml_list, :map, virtual: true, default: %{})
    field(:current_item, :map, virtual: true, default: nil)

    timestamps()
  end

  @required [:url]
  @optional [:next_guid]

  @doc """
  Builds a changeset based on the `struct` and `params`.
  """
  def changeset(struct, params \\ %{}) do
    struct
    |> cast(params, @required ++ @optional)
    |> validate_required(@required)
  end

  def update(%{id: id} = new_feed) do
    feed = Repo.get(__MODULE__, id)
    changeset = changeset(feed, Map.take(new_feed, @required ++ @optional))
    Repo.update(changeset)
    new_feed
  end

  def all(), do: MissionControlEx.Web.Repo.all(__MODULE__)

  def update_xml(feed, tries \\ 3)
  def update_xml(_, 0), do: {:error, "No xml found"}

  def update_xml(%__MODULE__{url: url} = feed, tries) do
    Logger.info("Attempting to fetch xml from url: #{url}")

    case MissionControlEx.MRSSFetcher.fetch_xml_as_string(url) do
      xml when is_bitstring(xml) and xml != "" -> %{feed | xml: xml}
      _ -> update_xml(feed, tries - 1)
    end
  end

  def xml_string_to_list(%{xml: xml_string} = feed) do
    xml_list =
      xmap(
        xml_string,
        items: [
          ~x"//item"l,
          guid: ~x"./guid/text()"s,
          title: ~x"./title/text()"s,
          episode: ~x"./media:episode/text()"s,
          duration_s: ~x"./media:asset-length/text()"f,
          description: ~x"./media:description/text()"s,
          content_url: ~x"./media:content/@url"s
        ]
      )

    %{feed | xml_list: xml_list.items, xml: nil}
  end

  def reset_feed(feed), do: %{feed | next_guid: nil}

  def get_xml(feed) do
    update_xml(feed)
    |> xml_string_to_list
  end

  def get_next_item(%__MODULE__{} = feed) do
    get_xml(feed)
    |> do_get_next_item
  end

  def get_next_item(id), do: Repo.get(__MODULE__, id) |> get_next_item

  defp do_get_next_item(%{next_guid: @done} = feed), do: %{feed | current_item: nil, xml_list: []}

  defp do_get_next_item(%{next_guid: nil, xml_list: [h | t]} = feed),
    do: set_next_guid(%{feed | current_item: h, xml_list: t})

  defp do_get_next_item(%{next_guid: next_guid, xml_list: [%{guid: next_guid} = h | t]} = feed),
    do: set_next_guid(%{feed | current_item: h, xml_list: t})

  defp do_get_next_item(%{next_guid: _, xml_list: [_ | t]} = feed),
    do: do_get_next_item(%{feed | xml_list: t})

  defp set_next_guid(%{xml_list: [h | _]} = feed), do: %{feed | next_guid: h.guid}
  defp set_next_guid(%{xml_list: []} = feed), do: %{feed | next_guid: @done}
end
