defmodule MissionControlEx.Web.Manifest do
  use MissionControlEx.Web, :model
  alias MissionControlEx.Web.Asset
  alias Ecto.Multi

  @delete_manifest_metadata %{
    "manifest_id" => "__delete__",
    "manifest_name" => "__delete__",
    "manifest_order_seq" => "__delete__"
  }
  @default_headers Asset.__schema__(:fields) -- [:id, :inserted_at, :updated_at, :metadata]

  schema "manifests" do
    field(:status, :string, default: "waiting")
    field(:name, :string)
    field(:manifest, :string)

    many_to_many(
      :assets,
      MissionControlEx.Web.Asset,
      join_through: "manifests_assets",
      on_replace: :delete
    )

    timestamps()
  end

  def get_by_name(name), do: Repo.get_by(__MODULE__, %{name: name})

  def new_manifest(changeset) do
    Repo.transaction(fn ->
      with {:ok, manifest} <- Repo.insert(changeset),
           {:ok, assets} <- create_transcode_jobs(manifest),
           twitch_stream <- create_twitch_stream(assets, manifest) do
        manifest
      else
        {:error, error} -> Repo.rollback(error)
      end
    end)
  end

  def load(%{id: id}), do: load(id)

  def load(manifest_id) do
    Repo.one(
      from(
        m in __MODULE__,
        where: m.id == ^manifest_id,
        left_join: assets in assoc(m, :assets),
        preload: [assets: assets],
        order_by: assets.id
      )
    )
    |> add_progress
  end

  def add_progress(%{assets: assets, status: "complete"} = manifest),
    do: Map.put(manifest, :progress, "#{length(assets)}/#{length(assets)}")

  def add_progress(%{assets: assets, status: status} = manifest) do
    complete = Enum.count(assets, fn asset -> asset.status == "complete" end)

    manifest =
      case complete == length(assets) do
        true -> Repo.update!(changeset(manifest, %{"status" => "complete"}))
        _ -> manifest
      end

    Map.put(manifest, :progress, "#{complete}/#{length(assets)}")
  end

  def add_assocs(_manifest, rows) when rows == nil or rows == [], do: []

  def add_assocs(manifest, rows) do
    # TODO: Use Ecto Multi for adding manifest assocs
    rows
    |> Asset.get_asset_from_rows()
    |> Repo.preload(:manifests)
    |> Enum.map(fn asset ->
         row_index = Enum.find_index(rows, &Asset.same?(asset, &1))

         manifest_metadata = %{
           "manifest_id" => manifest.id,
           "manifest_name" => manifest.name,
           "manifest_order_seq" => row_index
         }

         row = Enum.find(rows, asset, &Asset.same?(asset, &1))
         new_metadata = Map.merge(row.metadata, manifest_metadata)
         row = Map.put(row, :metadata, new_metadata)
         manifest_assoc = add_manifest_assoc(asset, manifest)
         update_manifest_assoc(asset, row, manifest_assoc)
       end)
  end

  def remove_old_assocs(manifest, fresh_rows) do
    stale_assets =
      load(manifest.id)
      |> Map.get(:assets)
      |> take_old_assets(fresh_rows)
      |> Repo.preload(:manifests)

    stale_assets
    |> Enum.map(fn asset -> {asset, @delete_manifest_metadata} end)
    |> Asset.update_metafields()

    # TODO: append Multi transactions
    stale_assets
    |> Enum.reduce(Multi.new(), fn asset, multi ->
         manifest_assoc_change =
           Asset.changeset(asset)
           |> put_assoc(:manifests, remove_manifest_assoc(asset, manifest))

         Multi.update(multi, Utils.random_string(64), manifest_assoc_change)
       end)
    |> Repo.transaction()
  end

  defp take_old_assets(stale_assets, fresh_rows) do
    Enum.reject(stale_assets, fn stale_asset ->
      Enum.any?(fresh_rows, &Asset.same?(&1, stale_asset))
    end)
  end

  defp update_manifest_assoc(asset, params, manifest_assoc_changes) do
    asset
    |> Asset.changeset(params)
    |> put_assoc(:manifests, manifest_assoc_changes)
    |> Repo.update!()
  end

  def create_transcode_jobs(
        %MissionControlEx.Web.Manifest{manifest: manifest_string, name: name} = manifest
      ) do
    {:ok, pid} = StringIO.open(manifest_string)
    pid |> IO.stream(:line) |> GenerateTranscodeJobs.run(manifest)
  end

  def create_twitch_stream(assets, manifest) do
    Repo.insert!(%MissionControlEx.Web.StreamSchedule{
      state: "waiting",
      name: manifest.name
    })
  end

  defp add_manifest_assoc(asset, manifest),
    do: Enum.uniq_by(asset.manifests ++ [manifest], & &1.id)

  defp remove_manifest_assoc(asset, manifest),
    do: Enum.filter(asset.manifests, &(&1.id != manifest.id))

  def manifest_to_csv(id) do
    manifest = load(id)

    metadata_headers =
      manifest.assets
      |> Enum.map(& &1.metadata)
      |> Utils.get_keys()

    headers = @default_headers ++ metadata_headers

    csv =
      manifest.assets
      |> Enum.map(&Map.merge(&1, &1.metadata || %{}))
      |> Enum.map(&Map.delete(&1, :metadata))
      |> Enum.map(&Map.update(&1, :commercial_breaks_ms, "[]", fn val -> Poison.encode!(val) end))
      |> Enum.map(&Map.take(&1, headers))
      |> CSV.encode(headers: headers)
      |> Enum.to_list()
      |> to_string

    {manifest, csv}
  end

  def csv_to_manifest(manifest, params) do
    manifest_params = csv_into_params(params)
    changeset(manifest, manifest_params)
  end

  def csv_into_params(%{"csv" => %Plug.Upload{path: file_path}} = params),
    do: Map.put(params, "manifest", File.read!(file_path))

  def csv_into_params(params), do: params

  @doc """
  Builds a changeset based on the `struct` and `params`.
  """
  def changeset(struct, params \\ %{}) do
    struct
    |> cast(params, [:status, :name, :manifest])
    |> validate_required([:name])
    |> unique_constraint(:name)
  end
end
