defmodule MissionControlEx.Web.Schedule do
  use MissionControlEx.Web, :model
  alias MissionControlEx.Web.{Asset, Event, EventChunk}

  defmodule ScheduleInvalidError do
    defexception [:message, :invalid_rows]

    def exception(message: msg) do
      %ScheduleInvalidError{message: msg}
    end

    def exception(invalid_rows: invalid_rows) do
      msg = "Invalid Rows"

      row_errors =
        Enum.map(invalid_rows, fn {{_false, row_msg}, row_number} ->
          "Row #{row_number}: #{row_msg}"
        end)

      %ScheduleInvalidError{message: msg, invalid_rows: row_errors}
    end
  end

  defmodule(
    Schedule.ScheduleMarker,
    do:
      defstruct([
        :schedule_block_length,
        :asset,
        :commercial_asset,
        marker_group: nil,
        ad_count: 0
      ])
  )

  alias Schedule.ScheduleMarker
  @schedule_headers ["schedule_block_length", "asset_path", "commercial_path"]
  @max_commercial_run_count 12

  def validate(rows, s3_path_to_asset) do
    {:ok, rows} = contains_headers?(rows)
    invalid_rows = get_invalid_rows(rows, s3_path_to_asset)

    case length(invalid_rows) do
      0 -> :ok
      _ -> raise ScheduleInvalidError, invalid_rows: invalid_rows
    end
  end

  def contains_headers?([%{} = row | _] = rows) do
    case Enum.all?(@schedule_headers, &Map.has_key?(row, &1)) do
      true ->
        {:ok, rows}

      _ ->
        raise ScheduleInvalidError,
          message: "missing required headers: #{Enum.join(@schedule_headers, ",")}"
    end
  end

  def contains_headers?([]), do: {:ok, []}

  def contains_headers?(_),
    do:
      raise(
        ScheduleInvalidError,
        message: "missing required headers: #{Enum.join(@schedule_headers, ",")}"
      )

  def get_invalid_rows(rows, s3_path_to_asset) do
    rows
    |> Enum.map(&validate_row(&1, s3_path_to_asset))
    |> Enum.with_index(1)
    |> Enum.reject(fn {{valid?, _err}, _row_num} -> valid? end)
  end

  def validate_row(
        %{
          "schedule_block_length" => schedule_block_length,
          "asset_path" => asset_path,
          "commercial_path" => commercial_path
        },
        s3_path_to_asset
      ),
      do: valid_row?(schedule_block_length, asset_path, commercial_path, s3_path_to_asset)

  def valid_row?(schedule_block_length, asset_path, commercial_path, s3_path_to_asset) do
    {asset_valid?, asset_err} = path_exists?(schedule_block_length, asset_path, s3_path_to_asset)

    {commercial_valid?, commercial_err} =
      path_exists?(schedule_block_length, commercial_path, s3_path_to_asset)

    {asset_valid? && commercial_valid?, "#{asset_err};#{commercial_err}"}
  end

  def path_exists?(block_length, "suspend", s3_path_to_asset) do
    case convert_to_seconds(block_length) do
      nil ->
        {false, "Invalid Block Length for Suspend: #{block_length} must be in hh:mm:ss format"}

      _ ->
        {true, ""}
    end
  end

  def path_exists?(_block_length, path, s3_path_to_asset) when path not in ["", "suspend"] do
    case s3_path_to_asset[path] do
      nil -> {false, "#{path} does not exist or had not finished transcoding"}
      _ -> {true, ""}
    end
  end

  def path_exists?(path, _, _), do: {true, ""}

  def to_event_list(rows) do
    s3_path_to_asset = s3_path_to_asset(rows)

    :ok = validate(rows, s3_path_to_asset)

    events =
      rows
      |> Enum.map(&to_schedule_marker(&1, s3_path_to_asset))
      |> assign_marker_groups
      |> Enum.chunk_by(& &1.marker_group)
      |> pad_marker_groups(s3_path_to_asset)
      |> List.flatten()
      |> Enum.with_index(1)
      |> Enum.map(&convert_to_events/1)
      |> Enum.map(&EventChunk.make_event_chunk/1)
  end

  def convert_to_events(
        {
          %ScheduleMarker{asset: :suspend, schedule_block_length: schedule_block_length},
          row_number
        }
      ),
      do: [Event.suspend(nil, "#{row_number}:suspend", schedule_block_length * 1000)]

  def convert_to_events(
        {
          %ScheduleMarker{asset: nil, commercial_asset: %Asset{}, ad_count: ad_count} = marker,
          row_number
        }
      ),
      do: commercial_event(marker, ad_count, row_number)

  def convert_to_events(
        {
          %ScheduleMarker{asset: %Asset{commercial_breaks_ms: []} = asset, ad_count: ad_count} =
            marker,
          row_number
        }
      ) do
    [
      Event.play_asset(nil, "#{row_number}:play", asset, 0, asset.metadata)
      | commercial_event(marker, ad_count, row_number)
    ]
  end

  def convert_to_events(
        {
          %ScheduleMarker{
            asset: %Asset{commercial_breaks_ms: commercial_breaks_ms} = asset,
            ad_count: ad_count
          } = marker,
          row_number
        }
      ) do
    commercial_ad_counts =
      allocate_ad_counts(length(commercial_breaks_ms) + 1, ad_count, row_number)

    Enum.reduce(0..length(commercial_breaks_ms), [], fn x, acc ->
      acc ++
        [
          Event.play_asset(nil, "#{row_number}:play", asset, x, asset.metadata)
          | commercial_event(marker, Enum.at(commercial_ad_counts, x), row_number)
        ]
    end)
  end

  def make_event_chunks(events), do: %EventChunk{events: events}

  def commercial_event(marker, 0, row_number), do: []

  def commercial_event(marker, ad_count, row_number)
      when ad_count * 30_000 > @max_commercial_run_count * 30_000,
      do:
        raise(
          ScheduleInvalidError,
          message: "Commerical length is greater than 6 mins for row #{row_number}"
        )

  def commercial_event(%{commercial_asset: commercial_asset} = marker, ad_count, row_number) do
    [
      Event.play_commercial(
        nil,
        "#{row_number}:commercial",
        commercial_asset,
        ad_count * 30_000 + 4_000
      )
    ]
  end

  def to_schedule_marker(
        %{
          "schedule_block_length" => schedule_block_length,
          "asset_path" => asset_path,
          "commercial_path" => commercial_path
        },
        s3_path_to_asset
      ) do
    asset = if asset_path == "suspend", do: :suspend, else: s3_path_to_asset[asset_path]

    %ScheduleMarker{
      schedule_block_length: convert_to_seconds(schedule_block_length),
      asset: asset,
      commercial_asset: s3_path_to_asset[commercial_path]
    }
  end

  def allocate_ad_counts(ad_breaks, 0, _) when ad_breaks > 0,
    do: Enum.map(1..ad_breaks, &(&1 * 0))

  def allocate_ad_counts(0, ad_count, row) when ad_count > 0,
    do:
      raise(
        ScheduleInvalidError,
        message:
          "Row #{row}: not enough ad breaks to allocate #{ad_count} ad-count, (max of 6 min per ad break)"
      )

  def allocate_ad_counts(0, 0, _row), do: []

  def allocate_ad_counts(ad_breaks, ad_count, row_num) do
    ads_in_break = Float.ceil(ad_count / ad_breaks) |> round
    ads_in_break = Enum.min([12, ads_in_break])
    [ads_in_break | allocate_ad_counts(ad_breaks - 1, ad_count - ads_in_break, row_num)]
  end

  def assign_marker_groups(schedule_chunks, accum \\ [])
  def assign_marker_groups([], accum), do: Enum.reverse(accum)
  def assign_marker_groups([h | t], []), do: assign_marker_groups(t, [%{h | marker_group: 0}])

  def assign_marker_groups(
        [%{schedule_block_length: nil} = h | t],
        [%{marker_group: marker_group} | _] = accum
      ) do
    assign_marker_groups(t, [%{h | marker_group: marker_group} | accum])
  end

  def assign_marker_groups(
        [%{schedule_block_length: schedule_block_length} = h | t],
        [%{marker_group: marker_group} | _] = accum
      ) do
    assign_marker_groups(t, [%{h | marker_group: marker_group + 1} | accum])
  end

  def pad_marker_groups(marker_groups, s3_path_to_asset, ad_budget \\ 0)
  def pad_marker_groups([], _, _), do: []

  def pad_marker_groups([marker_group | t], s3_path_to_asset, ad_budget) do
    {marker_group, ad_budget} = pad_marker_group(marker_group, s3_path_to_asset, ad_budget)
    [marker_group | pad_marker_groups(t, s3_path_to_asset, ad_budget)]
  end

  def pad_marker_group(marker_group, s3_path_to_asset, ad_budget) do
    [%{schedule_block_length: schedule_block_length} | _] = marker_group
    schedule_block_length = schedule_block_length * 1000

    case compute_duration(marker_group) do
      duration when duration < schedule_block_length + ad_budget ->
        do_pad_marker_group(marker_group, s3_path_to_asset, ad_budget)

      duration ->
        {marker_group, schedule_block_length + ad_budget - duration}
    end
  end

  @doc """
    Padding strategy is to increase the lowest marker by 30 seconds.
  """
  def do_pad_marker_group(marker_group, s3_path_to_asset, ad_budget) do
    shortest_ad_break =
      marker_group
      |> Enum.map(& &1.ad_count)
      |> Enum.min()

    first_shortest_ad_break_index =
      marker_group
      |> Enum.find_index(&(&1.ad_count == shortest_ad_break))

    List.update_at(marker_group, first_shortest_ad_break_index, fn marker ->
      %{marker | ad_count: marker.ad_count + 1}
    end)
    |> pad_marker_group(s3_path_to_asset, ad_budget)
  end

  def compute_duration([]), do: 0

  def compute_duration([%{asset: :suspend, schedule_block_length: schedule_block_length}]),
    do: schedule_block_length * 1000

  def compute_duration([%{ad_count: 0, asset: %{duration: duration}} | t]),
    do: duration + compute_duration(t)

  def compute_duration([%{ad_count: ad_count, asset: nil} | t]),
    do: 4_000 + ad_count * 30_000 + compute_duration(t)

  def compute_duration([
        %{ad_count: ad_count, asset: %{duration: duration, commercial_breaks_ms: []}} | t
      ]),
      do: 4_000 + ad_count * 30_000 + duration + compute_duration(t)

  def compute_duration([
        %{
          ad_count: ad_count,
          asset: %{duration: duration, commercial_breaks_ms: [_ | _] = commercial_breaks_ms}
        }
        | t
      ])
      when rem(ad_count, length(commercial_breaks_ms)) != 0,
      do: 8_000 + ad_count * 30_000 + duration + compute_duration(t)

  def compute_duration([
        %{
          ad_count: ad_count,
          asset: %{duration: duration, commercial_breaks_ms: [_ | _] = commercial_breaks_ms}
        }
        | t
      ]),
      do: 4_000 + ad_count * 30_000 + duration + compute_duration(t)

  def s3_path_to_asset(rows) do
    asset_paths = Enum.flat_map(rows, &get_asset_paths/1)

    Repo.all(
      from(
        a in Asset,
        where: a.s3_path in ^asset_paths,
        where: a.status == "complete",
        select: {a.s3_path, a}
      )
    )
    |> Enum.into(%{})
  end

  def get_asset_paths(%{"asset_path" => asset_path, "commercial_path" => commercial_path}),
    do: [asset_path, commercial_path]

  def get_asset_paths(%{"asset_path" => asset_path}), do: [asset_path]
  def get_asset_paths(%{"commercial_path" => commercial_path}), do: [commercial_path]
  def get_asset_paths(%{}), do: []

  def convert_to_seconds(ts) do
    case String.split(ts, ":") do
      [h, m, s] ->
        {hours, _} = Integer.parse(h)
        {minutes, _} = Integer.parse(m)
        {seconds, _} = Integer.parse(s)
        hours * 3600 + minutes * 60 + seconds

      _ ->
        nil
    end
  end
end
