defmodule MissionControlEx.Web.ScheduleManager do
  use MissionControlEx.Web, :model
  use Retry
  alias MissionControlEx.Web.{Channel, StreamSchedule, Event, EventChunk}
  require Logger
  {:ok, endpoint} = Application.fetch_env(:mission_control_ex, :event_api_url)
  @api_endpoint endpoint

  @derive {
            Poison.Encoder,
            only: [
              :id,
              :stream_schedules,
              :cron,
              :last_started_at,
              :last_live_at,
              :status,
              :events,
              :is_recurring,
              :start_times
            ]
          }

  schema "airings" do
    belongs_to(:channel, Channel)

    has_many(
      :stream_schedules,
      StreamSchedule,
      foreign_key: :schedule_manager_id,
      on_replace: :delete
    )

    field(:cron, Crontab.CronExpression.Ecto.Type)
    field(:last_started_at, :naive_datetime)
    field(:last_live_at, :naive_datetime)
    field(:status, :string)
    field(:is_recurring, :boolean, default: false)
    has_many(:events, through: [:stream_schedules, :events])

    field(:start_times, {:array, :string}, default: [], virtual: true)

    timestamps()
  end

  def changeset(struct, params \\ %{})

  def changeset(struct, params_in) do
    with params <- convert_cron_param(params_in) do
      struct
      |> cast(params, [
           :channel_id,
           :cron,
           :last_started_at,
           :status,
           :updated_at,
           :last_live_at,
           :is_recurring
         ])
      |> validate_required([])
    end
  end

  def force_start(schedule_manager) do
    __MODULE__.update(schedule_manager, %{status: "__FORCE_START__"})
  end

  def start_from_chunk(schedule_manager, chunk_id) do
    case find_schedule_with_chunk_id(schedule_manager, chunk_id) do
      %StreamSchedule{} = schedule ->
        updated_schedule = StreamSchedule.start_from(schedule, chunk_id)

        schedule_manager
        |> with_stream_schedule(updated_schedule)

      nil ->
        raise(
          "Unable to find chunk_id #{chunk_id} in schedule_manager with id: #{schedule_manager.id}"
        )
    end
  end

  def find_schedule_with_chunk_id(
        %{stream_schedules: stream_schedules} = schedule_manager,
        chunk_id
      ) do
    Enum.find(stream_schedules, &StreamSchedule.has_chunk(&1, chunk_id))
  end

  def html_safe(%{cron: nil} = airing), do: airing

  def html_safe(airing) do
    %{airing | cron: Crontab.CronExpression.Composer.compose(airing.cron)}
  end

  def convert_cron_param(%{cron: cron_string} = params),
    do: %{params | cron: convert_cron(cron_string)}

  def convert_cron_param(%{"cron" => cron_string} = params),
    do: %{params | "cron" => convert_cron(cron_string)}

  def convert_cron_param(params), do: params

  def convert_cron(cron_string) do
    {:ok, cron} =
      case length(String.split(cron_string, " ")) do
        7 -> Crontab.CronExpression.Parser.parse(cron_string, true)
        _ -> Crontab.CronExpression.Parser.parse(cron_string)
      end

    cron
  end

  def touch(airing) do
    Repo.update!(
      changeset(airing, %{
        updated_at: NaiveDateTime.utc_now(),
        last_live_at: NaiveDateTime.utc_now()
      })
    )
  end

  def update(airing_id, %{} = changes) do
    with airing <- load(airing_id) do
      changeset(airing, changes)
      |> Repo.update()
    end
  end

  def update!(airing_id, changes) do
    {:ok, airing} = __MODULE__.update(airing_id, changes)
    airing
  end

  def get_schedules(%{id: id} = airing) do
    Repo.all(
      from(
        s in StreamSchedule,
        join: sm in assoc(s, :airings),
        where: sm.id == ^id,
        select: s
      )
    )
    |> Enum.map(&StreamSchedule.load/1)
  end

  def get_schedules(id), do: get_schedules(Repo.get_by(__MODULE__, id: id))

  def get_default_schedule(%{id: id} = airing), do: get_default_schedule(id)

  def get_default_schedule(id) do
    Repo.one(
      from(
        s in StreamSchedule,
        join: sm in assoc(s, :airings),
        where: sm.id == ^id,
        where: s.default_schedule == true,
        select: s
      )
    )
    |> StreamSchedule.load()
  end

  def load(%__MODULE__{id: nil} = manager), do: manager
  def load(%{id: id}), do: load(id)

  def load(airing_id) do
    event_limit_query = from(e in Event, order_by: [desc: :inserted_at], limit: 300)

    Repo.one(
      from(
        sm in __MODULE__,
        where: sm.id == ^airing_id,
        preload: [events: ^event_limit_query]
      )
    )
    |> reverse_events
    |> load_channel
    |> load_stream_schedules
  end

  def load_all(schedule_managers), do: Enum.map(schedule_managers, fn %{id: id} -> load(id) end)
  def reverse_events(%{events: events} = sm), do: %{sm | events: Enum.reverse(events)}

  def load_channel(airing), do: %{airing | channel: Channel.get_channel(airing.channel_id)}
  def load_stream_schedules(airing), do: %{airing | stream_schedules: get_schedules(airing)}

  def get_start_time(%{cron: nil}), do: nil
  def get_start_time(%{status: "recovering"}), do: NaiveDateTime.utc_now()

  def get_start_time(%{status: "streaming", events: events}) when length(events) > 0 do
    event = Enum.fetch!(events, -1)
    Timex.add(event.inserted_at, Timex.Duration.from_milliseconds(Event.duration(event)))
  end

  def get_start_time(%{cron: cron, last_started_at: nil, inserted_at: inserted_at}),
    do: Crontab.Scheduler.get_next_run_date!(cron, inserted_at, 100_000_000)

  def get_start_time(%{cron: cron, last_started_at: last_started_at}),
    do: Crontab.Scheduler.get_next_run_date!(cron, last_started_at, 100_000_000)

  def get_start_times(%{cron: cron, is_recurring: true} = schedule_manager, amount) do
    start_time = get_start_time(%{schedule_manager | status: "waiting"})

    {start_times, _} =
      Enum.map_reduce(0..amount, start_time, fn x, acc ->
        next_run =
          Crontab.Scheduler.get_next_run_date!(cron, NaiveDateTime.add(acc, 10), 10_000_000)

        {acc, next_run}
      end)

    %{schedule_manager | start_times: start_times}
  end

  def get_start_times(%{is_recurring: false, status: "waiting"} = manager, _),
    do: %{manager | start_times: [get_start_time(manager)]}

  def get_start_times(%{is_recurring: false} = manager, _), do: %{manager | start_times: []}

  def get_next_chunk(manager, last_chunk_id \\ nil) do
    # TODO: Make this less scary (take_while not end event?)
    unplayed_chunks =
      manager
      |> generate_event_chunk_stream
      |> Stream.take(100)
      |> Enum.to_list()
      |> Enum.map(fn %{event_chunk: chunk} -> chunk end)

    case Enum.find_index(unplayed_chunks, fn %{id: id} ->
           id == last_chunk_id || last_chunk_id == "recovered_chunk"
         end) do
      nil ->
        Enum.at(unplayed_chunks, 0)

      index ->
        Enum.at(unplayed_chunks, index + 1) || {:error, :stream_done_or_out_of_sync}
    end
  end

  def generate_http_chunk_stream(%{id: id} = schedule_manager) do
    url = "#{@api_endpoint}/schedule_managers/#{id}/get_next_chunk"

    Stream.resource(fn -> {url, ""} end, &reduce_next_http_chunk(Mix.env(), &1), fn _ -> :ok end)
  end

  # Catch for mix_env="test"
  defp reduce_next_http_chunk(:test, {url, last_id}) do
    # Get ID from URL
    {id, _} =
      url
      |> String.split("/")
      |> Enum.at(-2)
      |> Integer.parse()

    schedule_manager = load(id)

    case get_next_chunk(schedule_manager, last_id) do
      {:error, _} -> {:halt, :ok}
      chunk -> {[chunk], {url, chunk.id}}
    end
  end

  defp reduce_next_http_chunk(_, {url, _last_id} = acc) do
    with chunk <- fetch_next_chunk(acc) do
      {[chunk], {url, chunk.id}}
    end
  end

  def fetch_next_chunk({url, last_chunk_id}) do
    retry_while with: lin_backoff(250, 2) |> expiry(10_000) do
      try do
        value =
          HTTPoison.get!(
            url,
            [{"Accept", "application/json"}],
            params: %{
              last_chunk_id: last_chunk_id
            }
          )
          |> Map.get(:body)
          |> Poison.decode!(as: %EventChunk{events: [%Event{}]})

        {:halt, value}
      rescue
        error ->
          {:cont, {:fetch_next_chunk_failed, error}}
      end
    end
  end

  def persist_stream_launched(schedule_manager) do
    get_default_schedule(schedule_manager)
    |> Map.get(:id)
    |> Event.launch_stream("launch")
    |> Repo.insert!()

    schedule_manager
  end

  @doc """
    Generates an event stream from a schedule manager
  """
  def generate_event_chunk_stream(schedule_manager, time \\ nil)

  def generate_event_chunk_stream(schedule_manager, time) do
    Stream.resource(
      fn ->
        %StreamManagerBlock{
          schedule_manager: schedule_manager,
          time_after_events: time || get_start_time(schedule_manager)
        }
      end,
      &get_next_event_chunk/1,
      fn _ -> :ok end
    )
  end

  def generate_event_stream(schedule_manager, time \\ nil) do
    start_time = time || get_start_time(schedule_manager)

    fake_manager_block = %StreamManagerBlock{
      schedule_manager: schedule_manager,
      time_after_events: start_time,
      chunk_index: 0
    }

    generate_event_chunk_stream(schedule_manager, start_time)
    |> Stream.transform(fake_manager_block, fn event_chunk_block, fake_manager_block ->
         StreamEventChunkBlock.to_stream_event_blocks(event_chunk_block, fake_manager_block)
       end)
  end

  defp get_next_event_chunk(
         %StreamManagerBlock{
           schedule_manager: schedule_manager,
           time_after_events: schedule_manager_time
         } = manager_block
       ) do
    generate_event_chunk_at(manager_block)
  end

  defp generate_event_chunk_at(
         %StreamManagerBlock{
           schedule_manager: %{status: "recovering", last_live_at: nil} = schedule_manager
         } = block
       ),
       do:
         generate_event_chunk_at(%{
           block
           | schedule_manager: %{schedule_manager | status: "streaming"}
         })

  defp generate_event_chunk_at(
         %StreamManagerBlock{
           schedule_manager: %{status: "recovering"} = schedule_manager,
           time_after_events: schedule_manager_time
         } = manager_block
       ) do
    schedule_manager = %{schedule_manager | status: "streaming"}
    recovery_events = generate_recovery_events(schedule_manager)
    %{stream_schedules: [stream_schedule | _]} = schedule_manager

    event_chunk =
      EventChunk.make_event_chunk(recovery_events)
      |> EventChunk.link_schedule(stream_schedule)
      |> Map.put(:id, "recovered_chunk")

    manager_event_tuple(schedule_manager, schedule_manager_time, stream_schedule, event_chunk)
  end

  defp generate_event_chunk_at(%StreamManagerBlock{
         schedule_manager: schedule_manager,
         time_after_events: schedule_manager_time
       }) do
    schedule_at =
      schedule_manager
      |> schedule_at(schedule_manager_time)

    {stream_schedule, event_chunk} = StreamSchedule.get_next_event_chunk(schedule_at)

    schedule_manager =
      schedule_manager
      |> with_stream_schedule(stream_schedule)
      |> with_events(event_chunk.events)

    manager_event_tuple(schedule_manager, schedule_manager_time, stream_schedule, event_chunk)
  end

  def manager_event_tuple(schedule_manager, schedule_manager_time, stream_schedule, event_chunk) do
    after_events_time = get_time_after(schedule_manager, schedule_manager_time, event_chunk)

    event_chunk = EventChunk.project_inserted_at(event_chunk, schedule_manager_time)

    manager_block = %StreamManagerBlock{
      schedule_manager: schedule_manager,
      time_after_events: after_events_time
    }

    event_chunk_block = %StreamEventChunkBlock{
      schedule_manager: schedule_manager,
      stream_schedule: stream_schedule,
      event_chunk: event_chunk
    }

    {[event_chunk_block], manager_block}
  end

  # Get next run time if end_stream
  def get_time_after(schedule_manager, schedule_manager_time, %{
        events: [%{type: "end_stream", data: %{"recurring" => true}} | _]
      }) do
    Crontab.Scheduler.get_next_run_date!(schedule_manager.cron, schedule_manager_time, 1_000_000)
  end

  # Add event durations to get time after
  def get_time_after(schedule_manager, schedule_manager_time, event_chunk) do
    Timex.add(
      schedule_manager_time,
      Timex.Duration.from_milliseconds(Event.duration(event_chunk.events))
    )
  end

  def with_events(schedule_manager, events) do
    events = Enum.reverse(events)
    %{schedule_manager | events: schedule_manager.events ++ events}
  end

  defp generate_recovery_events(%{last_live_at: last_live_at} = schedule_manager)
       when last_live_at != nil do
    recovery_time = Timex.add(last_live_at, Timex.Duration.from_seconds(-30))
    unix_recovery_time = Timex.to_unix(recovery_time)

    recovery_events =
      schedule_manager
      |> Map.get(:events)
      |> Enum.reverse()
      |> Enum.take_while(fn event -> Timex.to_unix(Event.ended_at(event)) > unix_recovery_time end)
      |> Enum.reverse()
      |> Enum.map(&recover_event(&1, recovery_time))
      |> Enum.split_while(&(!is_video_event(&1)))
      |> join_on_black_buffer
  end

  defp generate_recovery_events(_), do: {:error, :no_last_live_at}

  @video_events [
    "play_commercial",
    "play_asset",
    "recovery_event",
    "play_remote_asset",
    "fallback_asset",
    "stream_black_buffer",
    "play_twitch_clip"
  ]
  defp is_video_event(%{type: type}) when type in @video_events, do: true
  defp is_video_event(_), do: false

  defp join_on_black_buffer({[], []}), do: []

  defp join_on_black_buffer(
         {
           non_video_events,
           [%{twitch_stream_id: twitch_stream_id} | _] = video_events
         }
       ) do
    non_video_events ++ [Event.stream_black_buffer(twitch_stream_id)] ++ video_events
  end

  defp join_on_black_buffer(
         {
           [%{twitch_stream_id: twitch_stream_id} | _] = non_video_events,
           video_events
         }
       ) do
    non_video_events ++ [Event.stream_black_buffer(twitch_stream_id)] ++ video_events
  end

  def recover_event(%{inserted_at: inserted_at} = event, recovery_time) do
    offset =
      [0, Timex.diff(recovery_time, inserted_at, :milliseconds)]
      |> Enum.max()

    %{data: data, twitch_stream_id: twitch_stream_id, ref: ref, type: type} =
      event
      |> update_in([Access.key!(:data), "start_time"], &add_offset_to_start_time(&1, offset))
      |> update_in([Access.key!(:data), "duration"], &subtract_offset_from_duration(&1, offset))
      |> put_in([Access.key!(:data), "recover"], true)

    # Put in new event so Repo.insert! works (won't work on loaded asset)
    %Event{data: data, twitch_stream_id: twitch_stream_id, ref: ref, type: type}
  end

  defp add_offset_to_start_time(nil, offset), do: offset
  defp add_offset_to_start_time(start_time, offset), do: start_time + offset

  defp subtract_offset_from_duration(nil, offset), do: nil
  defp subtract_offset_from_duration(duration, offset), do: duration - offset

  defp schedule_at(%{stream_schedules: stream_schedules}, schedule_manager_time) do
    {[default_schedule], schedules} =
      stream_schedules
      |> Enum.split_with(& &1.default_schedule)

    schedules
    |> Enum.filter(fn %{start_cron: cron, duration: duration} ->
         Crontab.Scheduler.get_previous_run_date!(cron, schedule_manager_time, 100_000_000)
         |> Timex.add(Timex.Duration.from_milliseconds(duration))
         |> Utils.time_compare?(schedule_manager_time)
       end)
    |> Enum.min_by(
         fn %{start_cron: cron} ->
           Crontab.Scheduler.get_previous_run_date!(cron, schedule_manager_time, 100_000_000)
         end,
         fn -> default_schedule end
       )
  end

  def with_stream_schedule(
        %{stream_schedules: stream_schedules} = schedule_manager,
        new_stream_schedule
      ) do
    %{
      schedule_manager
      | stream_schedules: [
          new_stream_schedule
          | Enum.filter(stream_schedules, fn %{id: id} -> id != new_stream_schedule.id end)
        ]
    }
  end

  def validate_has_default(changeset, %{"default_schedule_id" => id} = params), do: changeset

  def validate_has_default(changeset, _params),
    do: add_error(changeset, :stream_schedules, "must have default schedule")

  def is_valid?(%__MODULE__{stream_schedules: schedules, channel: %Channel{}} = manager),
    do: Enum.any?(schedules, & &1.default_schedule)

  def is_valid?(_), do: false

  def create_empty_manager(%StreamSchedule{} = ss) do
    {:ok, default_cron} = Crontab.CronExpression.Parser.parse("* * * * * *", true)

    %__MODULE__{
      cron: ss.start_cron || default_cron,
      status: "waiting",
      inserted_at: NaiveDateTime.utc_now(),
      stream_schedules: [ss],
      events: []
    }
  end
end
