defimpl Phoenix.HTML.Safe, for: MissionControlEx.Web.StreamSchedule do
  def to_iodata(data), do: data |> Poison.encode!() |> Plug.HTML.html_escape()
end

defmodule MissionControlEx.Web.StreamSchedule do
  use MissionControlEx.Web, :model
  alias Ecto.Multi
  alias MissionControlEx.Web.{Schedule, ScheduleManager, Event, EventChunk}

  @five_seconds 5_000

  @derive {
            Poison.Encoder,
            only: [
              :id,
              :name,
              :state,
              :events,
              :queued_event_chunks,
              :played_event_chunks,
              :duration,
              :start_cron,
              :schedule_manager_id,
              :default_schedule,
              :saved,
              :api_url,
              :api_events_before_replay,
              :api_num_days_to_fetch,
              :repeat,
              :stop_on_empty,
              :shuffle
            ]
          }
  schema "twitch_streams" do
    field(:name, :string)
    field(:duration, :integer)

    field(:state, :string, default: "waiting")

    field(:api_url, :string)
    field(:api_events_before_replay, :integer, default: 500)
    field(:api_num_days_to_fetch, :integer, default: 7)

    field(:event_chunks, EventChunkList, default: [])
    field(:queued_event_chunks, EventChunkList, default: [])
    field(:played_event_chunks, EventChunkList, default: [])
    field(:chunk_index, :integer)
    field(:current_event, :any, virtual: true)
    field(:repeat, :boolean)
    field(:shuffle, :boolean)
    field(:stop_on_empty, :boolean, default: true)

    field(:start_cron, Crontab.CronExpression.Ecto.Type)
    field(:default_schedule, :boolean, default: false)
    field(:last_run, :naive_datetime)
    field(:next_run_time, :naive_datetime, virtual: true)

    belongs_to(:airings, ScheduleManager, foreign_key: :schedule_manager_id)

    has_many(:events, MissionControlEx.Web.Event, foreign_key: :twitch_stream_id)

    many_to_many(:channels, MissionControlEx.Web.Channel, join_through: "airings")

    timestamps()
  end

  @required [:state, :name]
  @optional [
    :duration,
    :start_cron,
    :last_run,
    :schedule_manager_id,
    :default_schedule,
    :repeat,
    :shuffle,
    :chunk_index,
    :stop_on_empty,
    :api_url,
    :event_chunks,
    :queued_event_chunks,
    :played_event_chunks,
    :api_events_before_replay,
    :api_num_days_to_fetch
  ]

  @doc """
  Non-destructive chunk functions are handled by ScheduleWorker
  """
  def load_event_list_from_csv(schedule, csv),
    do: ScheduleWorker.load_event_list_from_csv(schedule, csv)

  def get_next_event_chunk(schedule), do: ScheduleWorker.get_next_event_chunk(schedule)

  def update_with_event(%{twitch_stream_id: nil} = target_event),
    do: {:error, "no stream schedule", target_event}

  def update_with_event(%{twitch_stream_id: ss_id} = target_event) do
    schedule = load(ss_id)
    do_update_with_event(schedule, target_event)
  end

  def has_chunk(schedule, chunk_id) do
    nil !=
      (schedule.event_chunks ++ schedule.played_event_chunks ++ schedule.queued_event_chunks)
      |> Enum.find(fn %{id: id} -> id == chunk_id end)
  end

  def start_from(schedule, chunk_id) do
    ScheduleWorker.do_reset_stream_schedule(schedule)
    |> fast_forward(chunk_id)
  end

  def fast_forward(schedule, chunk_id) do
    # Get next until on correct chunk
    Stream.resource(
      fn -> get_next_event_chunk(schedule) end,
      fn
        {schedule, %{id: id} = chunk} when id == chunk_id ->
          {:halt, schedule}

        {schedule, chunk} ->
          {[schedule], get_next_event_chunk(schedule)}
      end,
      fn _ -> :ok end
    )
    |> Enum.to_list()
    |> Enum.at(-1)
    |> Map.put(:chunk_index, nil)
    |> persist_changes
  end

  # Event is within current played_event_chunk
  def do_update_with_event(
        %{played_event_chunks: [%{id: chunk_id} | _]} = schedule,
        %{chunk_id: chunk_id, chunk_index: chunk_index} = event
      ) do
    %{schedule | chunk_index: chunk_index}
    |> persist_changes
  end

  # Event is on queued_event_chunks waiting to be popped
  def do_update_with_event(
        %{queued_event_chunks: [%{id: chunk_id} | _]} = schedule,
        %{chunk_id: chunk_id, chunk_index: chunk_index} = event
      ) do
    %{schedule | chunk_index: chunk_index}
    |> ScheduleWorker.pop_queued_event_chunks()
    |> persist_changes
  end

  # No more event chunks, reset or die
  def do_update_with_event(%{queued_event_chunks: []} = schedule, %{chunk_id: chunk_id} = event) do
    case get_next_event_chunk(schedule) do
      {done_schedule, %{id: chunk_id}} -> persist_changes(done_schedule)
      {updated_schedule, _next_chunk} -> {:noop, updated_schedule, event}
    end
  end

  def do_update_with_event(schedule, event) do
    {:noop, schedule, event}
  end

  def persist_changes(stream_schedule) do
    MissionControlEx.Web.StreamSchedule.update(stream_schedule.id, %{
      event_chunks: stream_schedule.event_chunks,
      queued_event_chunks: stream_schedule.queued_event_chunks,
      played_event_chunks: stream_schedule.played_event_chunks,
      chunk_index: stream_schedule.chunk_index
    })
  end

  @doc """
  Builds a changeset based on the `struct` and `params`.
  """
  def changeset(struct, params \\ %{})

  def changeset(struct, %{start_cron: cron_string} = changes) when is_bitstring(cron_string) do
    {:ok, cron} = Crontab.CronExpression.Parser.parse(cron_string)
    changeset(struct, %{changes | start_cron: cron})
  end

  def changeset(struct, params) do
    struct
    |> cast(params, @required ++ @optional)
    |> validate_required(@required)
    |> foreign_key_constraint(
         :schedule_manager_id,
         name: :twitch_streams_schedule_manager_id_fkey
       )
  end

  def get_stream(stream_id), do: Repo.get_by(__MODULE__, id: stream_id)

  def get_channels(%__MODULE__{id: id}), do: get_channels(id)

  def get_channels(twitch_stream_id) do
    Repo.all(
      from(
        a in ScheduleManager,
        join: tw in assoc(a, :stream_schedules),
        join: c in assoc(a, :channel),
        where: tw.id == ^twitch_stream_id,
        select: c
      )
    )
  end

  def get_live_channels(%__MODULE__{id: id}), do: get_live_channels(id)

  def get_live_channels(twitch_stream_id) do
    Repo.all(
      from(
        a in ScheduleManager,
        join: tw in assoc(a, :stream_schedules),
        join: c in assoc(a, :channel),
        where: a.status == "streaming",
        where: tw.id == ^twitch_stream_id,
        select: c
      )
    )
  end

  def load(%__MODULE__{id: id}), do: load(id)

  def load(twitch_stream_id) do
    event_limit_query = from(e in Event, order_by: [desc: :inserted_at], limit: 300)

    Repo.one(
      from(
        ss in __MODULE__,
        where: ss.id == ^twitch_stream_id,
        preload: [:airings, events: ^event_limit_query]
      )
    )

    # |> load_next_run_time
  end

  def reset_stream_schedule(%{id: id}), do: reset_stream_schedule(id)

  def reset_stream_schedule(stream_schedule_id) do
    load(stream_schedule_id)
    |> persist_changes
  end

  def load_next_run_time(%{start_cron: nil} = schedule),
    do: %{schedule | next_run_time: NaiveDateTime.utc_now()}

  def load_next_run_time(%{last_run: nil} = schedule),
    do: do_load_run_time(schedule, NaiveDateTime.utc_now())

  def load_next_run_time(%{last_run: last_run} = schedule),
    do: do_load_run_time(schedule, last_run)

  def load_next_run_time(%{inserted_at: inserted_at} = schedule),
    do: do_load_run_time(schedule, inserted_at)

  defp do_load_run_time(%{start_cron: cron} = schedule, time) do
    task = Task.async(fn -> Crontab.Scheduler.get_next_run_date(cron, time, 1_000_000) end)

    next_scheduled_time =
      case Task.yield(task, 2000) || Task.shutdown(task) do
        {:ok, {:ok, next_scheduled_time}} -> next_scheduled_time
        other -> nil
      end

    %{schedule | next_run_time: next_scheduled_time}
  end

  def create_default_schedule(
        _schedule_manager,
        %{"default_schedule_id" => _default_schedule} = airing_params
      ) do
    {:ok, nil}
  end

  def create_default_schedule(schedule_manager, _airing_params) do
    default_schedule = %{
      name: "default-#{schedule_manager.id}",
      state: "waiting",
      default_schedule: true,
      schedule_manager_id: schedule_manager.id
    }

    Repo.insert(changeset(%__MODULE__{}, default_schedule))
  end

  def html_safe(%{start_cron: nil} = stream_schedule), do: stream_schedule

  def html_safe(stream_schedule) do
    %{
      stream_schedule
      | start_cron: Crontab.CronExpression.Composer.compose(stream_schedule.start_cron)
    }
  end

  def end_stream(%{repeat: false, airings: schedule_manager} = stream_schedule),
    do: ScheduleManager.update(schedule_manager.id, %{status: "done"})

  def end_stream(
        %{repeat: true, stop_on_empty: true, airings: schedule_manager} = stream_schedule
      ),
      do: ScheduleManager.update(schedule_manager.id, %{status: "waiting"})

  def update(%__MODULE__{} = twitch_stream, changes) do
    changeset(twitch_stream, changes)
    |> Repo.update!()
  end

  def update(id, changes), do: MissionControlEx.Web.StreamSchedule.update(load(id), changes)

  def update_schedule_manager_assoc(
        schedule_manager,
        %{"schedules" => schedules, "default_schedule_id" => default_schedule_id} = params
      ) do
    %{stream_schedules: old_schedules} = Repo.preload(schedule_manager, :stream_schedules)
    stale_schedules = get_stale_schedules(schedules, old_schedules)

    Multi.new()
    |> remove_stale_schedule_assocs(stale_schedules)
    |> add_fresh_schedule_assocs(schedules, schedule_manager)
    |> update_default_schedule(default_schedule_id)
    |> Repo.transaction()
  end

  def update_schedule_manager_assoc(_, _), do: {:ok, []}

  def update_default_schedule(multi, default_schedule_id) do
    q = from(s in __MODULE__, where: s.id == ^default_schedule_id)

    Multi.update_all(
      multi,
      Utils.random_string(64),
      q,
      [set: [default_schedule: true]],
      returning: true
    )
  end

  def duplicate(stream_schedule) do
    values = Map.take(stream_schedule, @required ++ @optional)

    new_set =
      MissionControlEx.Web.StreamSchedule.changeset(%__MODULE__{}, %{
        values
        | name: values.name <> "_copy",
          schedule_manager_id: nil
      })

    Repo.insert!(new_set)
  end

  def get_stale_schedules(fresh_set, older_set) do
    Enum.reject(older_set, fn %{id: id_old_item} ->
      Enum.any?(fresh_set, fn {_index, %{"id" => id_fresh_item}} ->
        Utils.parse_int(id_fresh_item) === id_old_item
      end)
    end)
  end

  def remove_stale_schedule_assocs(multi, stale_schedules) do
    Enum.reduce(stale_schedules, multi, fn schedule, multi ->
      Multi.update(
        multi,
        Utils.random_string(64),
        changeset(schedule, %{schedule_manager_id: nil})
      )
    end)
  end

  def add_fresh_schedule_assocs(multi, schedules, schedule_manager) do
    q =
      Enum.reduce(schedules, __MODULE__, fn {_index, %{"id" => id}}, query ->
        from(q in query, or_where: [{:id, ^id}])
      end)

    Multi.update_all(
      multi,
      Utils.random_string(64),
      q,
      [set: [schedule_manager_id: schedule_manager.id, default_schedule: false]],
      returning: true
    )
  end

  def all_without_events() do
    ignored_fields = [:events, :queued_event_chunks, :played_event_chunks]
    only = __MODULE__.__schema__(:fields) -- ignored_fields

    Repo.all(from(ss in __MODULE__, select: ^only))
  end
end
