defmodule MissionControlEx.Web.Router do
  use MissionControlEx.Web, :router
  use Plug.ErrorHandler

  pipeline :browser do
    plug(:accepts, ["html", "json"])
    plug(:fetch_session)
    plug(:fetch_flash)
    # plug :protect_from_forgery
    plug(:put_secure_browser_headers)
  end

  pipeline :api do
    plug(:accepts, ["json"])
  end

  scope "/", MissionControlEx.Web do
    # Use the default browser stack
    pipe_through(:browser)

    scope "/auth" do
      get("/:provider", AuthController, :request)
      get("/:provider/callback", AuthController, :callback)
    end

    get("/", PageController, :index)

    post("/stream_schedules/:id/reset_stream", StreamScheduleController, :reset_stream)
    post("/stream_schedules/:id/duplicate", StreamScheduleController, :duplicate)
    resources("/stream_schedules", StreamScheduleController)

    get("/stream_schedules/:id/curator", StreamScheduleController, :curator)

    put(
      "/stream_schedules/:id/save_curated_changes",
      StreamScheduleController,
      :save_curated_changes
    )

    get(
      "/stream_schedules/:id/create_curated_chunk",
      StreamScheduleController,
      :create_curated_chunk
    )

    get("/stream_schedules/:id/editor", StreamScheduleController, :editor)
    get("/stream_schedules/:id/editor/assets", StreamScheduleController, :assets)
    put("/stream_schedules/:id/editor", StreamScheduleController, :update_schema)

    post("/streamer/:schedule_manager_id/stop", StreamerController, :stop)

    post("/manifests/s3/csv", ManifestController, :export_csv)
    get("/manifests/:id/csv", ManifestController, :export_manifest)
    get("/manifests/s3/assets", ManifestController, :s3_assets)
    get("/manifests/s3", ManifestController, :s3_manifest)
    resources("/manifests", ManifestController)

    post("/transcoder", CaptionsController, :create_captions)

    resources("/channels", ChannelController)

    resources("/schedule_managers", ScheduleManagerController)
    post("/schedule_managers/:id", ScheduleManagerController, :launch)
    get("/schedule_managers/:schedule_manager_id/preview", ScheduleManagerController, :preview)
    get("/schedule_managers/:id/get_next_chunk", ScheduleManagerController, :get_next_chunk)

    resources("/interstitials", InterstitialController)

    get("/assets/search/csv", AssetController, :export_csv)
    get("/assets/search", AssetController, :search)
    get("/assets/sources", AssetController, :search_source)
    get("/assets/sequence", AssetController, :sequence)
    post("/assets/sequence", AssetController, :make_sequence)
    resources("/assets", AssetController)

    post("/events/process", EventController, :process_and_persist_event)

    resources("/chat_commands", ChatCommandController)
  end

  # Other scopes may use custom stacks.
  # scope "/api", MissionControlEx do
  #   pipe_through :api
  # end
  defp handle_errors(conn, %{kind: kind, reason: reason, stack: stacktrace}) do
    conn =
      conn
      |> Plug.Conn.fetch_cookies()
      |> Plug.Conn.fetch_query_params()

    conn_data = %{
      "request" => %{
        "cookies" => conn.req_cookies,
        "url" => "#{conn.scheme}://#{conn.host}:#{conn.port}#{conn.request_path}",
        "user_ip" => conn.remote_ip |> Tuple.to_list() |> Enum.join("."),
        "headers" => Enum.into(conn.req_headers, %{}),
        "params" => conn.params,
        "method" => conn.method
      },
      "server" => %{
        "pid" => self()
      }
    }

    Rollbax.report(kind, reason, stacktrace, %{}, conn_data)
  end
end
