defmodule MigrateLegacyAssets do
  alias Flow
  alias MissionControlEx.Web.{Repo, Asset}
  alias Ecto.Multi
  @bucket "twitch-creative-video-repository"

  def run() do
    legacy_assets =
      Repo.all(Asset)
      |> Enum.filter(&String.contains?(&1.s3_path, "prod_final/"))

    with {:ok, _} <- tag_legacy_assets(legacy_assets),
         {:ok, _} <- migrate_s3_paths(legacy_assets),
         _ <- mass_copy_subs() do
      IO.inspect("SUCCESS")
    else
      error -> raise error
    end
  end

  def tag_legacy_assets(assets) do
    assets
    |> Enum.map(fn asset -> {asset, %{legacy_s3_path_asset: true}} end)
    |> Asset.update_metafields()
  end

  def migrate_s3_paths(assets) do
    assets
    |> Enum.reduce(Multi.new(), fn asset, multi ->
         new_path =
           get_new_path(asset)
           |> IO.inspect()

         Multi.update(
           multi,
           Utils.random_string(64),
           Asset.changeset(asset, %{s3_path: new_path})
         )
       end)
    |> Repo.transaction()
  end

  def mass_copy_subs() do
    Asset.get_assets_by_metafield("legacy_s3_path_asset")
    |> Flow.from_enumerable()
    |> Flow.partition(stages: 20)
    |> Flow.map(fn asset ->
         case asset.source_subtitles do
           "None_Required" ->
             :ok

           source_subtitles ->
             Transcode.copy_to_(source_subtitles, Transcode.s3_path(:subtitles, asset))
         end
       end)
    |> Enum.to_list()
    |> IO.inspect()
  end

  def set_to_waiting() do
    Asset.get_assets_by_metafield("legacy_s3_path_asset")
    |> Enum.reduce(Multi.new(), fn asset, multi ->
         Multi.update(
           multi,
           Utils.random_string(64),
           Asset.changeset(asset, %{status: "waiting"})
         )
       end)
    |> Repo.transaction()
  end

  def get_new_path(%{s3_path: old_path, source_movie: source_movie} = asset) do
    manifest_name =
      old_path
      |> String.split("/")
      |> Enum.at(1)

    GenerateTranscodeJobs.final_s3_path(%{name: manifest_name}, source_movie)
  end

  def get_legacy_path(%{source_movie: source_movie} = legacy_asset) do
    old_path =
      String.replace(source_movie, "prod_raw/", "prod_final/")
      |> String.replace(Utils.get_extname(source_movie), ".flv")
  end

  def revert() do
    with {:ok, _} <- revert_to_legacy() do
      IO.inspect("REVERT SUCCESS")
    else
      error -> raise error
    end
  end

  def revert_to_legacy() do
    Asset.get_assets_by_metafield("legacy_s3_path_asset")
    |> Enum.reduce(Multi.new(), fn asset, multi ->
         legacy_path =
           get_legacy_path(asset)
           |> IO.inspect()

         Multi.update(
           multi,
           Utils.random_string(64),
           Asset.changeset(asset, %{s3_path: legacy_path})
         )
       end)
    |> Repo.transaction()
  end
end

case System.get_env("MODE") do
  "revert" -> MigrateLegacyAssets.revert()
  "transcode" -> MigrateLegacyAssets.set_to_waiting()
  "copy" -> MigrateLegacyAssets.mass_copy_subs()
  _ -> MigrateLegacyAssets.run()
end
