defmodule MissionControlEx.Web.ApiStream do
  use HTTPoison.Base
  use Retry
  alias ExAws.S3
  require Utils
  require Logger

  def get_clips_and_cursor(url, cursor, num_days) do
    ConCache.get_or_store(:api_cache, {url, cursor}, fn ->
      do_get_clips_and_cursor(url, cursor, num_days)
    end)
  end

  def do_get_clips_and_cursor(url, cursor, num_days) do
    offset_url = get_offset_url(url, num_days)

    retry_while with: lin_backoff(250, 2) |> expiry(10_000) do
      try do
        %{"clips" => clips, "_cursor" => cursor} =
          HTTPoison.get!(offset_url <> "&cursor=#{cursor}", RequestUtils.headers_v5())
          |> RequestUtils.unpack_body_from_response()

        {:halt, {clips, {cursor, url}}}
      rescue
        error ->
          {:cont, {:get_clips_and_cursor_failed, error}}
      end
    end
    |> case do
         {:get_clips_and_cursor_failed, reason} ->
           raise "Get clips and cursor failed for reason #{reason}"

         {clips, {cursor, url}} ->
           {clips, {cursor, url}}
       end
  end

  defp get_offset_url(url, num_days) do
    start_time =
      DateTime.utc_now()
      |> Timex.shift(days: -(num_days + 1))
      |> format_time

    end_time =
      DateTime.utc_now()
      |> Timex.shift(days: -1)
      |> format_time

    # Hack to set time back 1 day, order of queries is significant, start time before end time
    url <> "&start_time=" <> start_time <> "&end_time=" <> end_time
  end

  defp format_time(time) do
    time =
      time
      |> Timex.format!("{ISO:Extended}")
      |> String.split("+")
      |> Enum.at(0)

    time <> "Z"
  end

  def get_unique_clips(url, amount, num_days, blacklist) do
    Stream.resource(
      fn -> {"", url} end,
      fn {cursor, url} -> get_clips_and_cursor(url, cursor, num_days) end,
      fn val -> :ok end
    )
    |> filter_clips(blacklist)
    |> Enum.take(amount)
  end

  defp filter_clips(stream, blacklist) do
    blacklist_ids = get_ids_from_events(blacklist)
    Stream.reject(stream, fn %{"slug" => clip_id} -> MapSet.member?(blacklist_ids, clip_id) end)
  end

  def get_ids_from_events(events) do
    Enum.flat_map(events, fn
      %{data: %{"id" => id}} -> [id]
      _ -> []
    end)
    |> MapSet.new()
  end
end
