defmodule MissionControlEx.Web.StreamScheduler.JobProducer do
  use GenStage
  alias MissionControlEx.Web.Repo
  alias MissionControlEx.Web.ScheduleManager
  import Ecto.Query
  @five_seconds 5_000
  def start_link do
    Task.async(&poll_queue/0)

    GenStage.start_link(
      MissionControlEx.Web.StreamScheduler.JobProducer,
      0,
      name: MissionControlEx.Web.StreamScheduler.JobProducer
    )
  end

  def poll_queue do
    :timer.sleep(@five_seconds)
    send(MissionControlEx.Web.StreamScheduler.JobProducer, :stream_enqueued)
    poll_queue
  end

  def init(counter) do
    {:producer, counter}
  end

  def handle_demand(demand, state) do
    limit = demand + state
    {:ok, {count, events}} = take(limit)
    {:noreply, ScheduleManager.load_all(events), limit - count}
  end

  def handle_info(:stream_enqueued, state) do
    limit = state
    {:ok, {count, events}} = take(limit)
    {:noreply, events, limit - count}
  end

  def take(0), do: {:ok, {0, []}}

  def take(limit) do
    Repo.transaction(fn ->
      schedule_managers =
        waiting_or_streaming_managers()
        |> Repo.all()
        |> Enum.filter(&scheduled_to_play/1)
        |> Enum.map(fn %{id: id} -> id end)
        |> get_managers_with_lock(limit)
        |> Repo.all()
        |> Enum.map(&update_stream_status/1)

      {length(schedule_managers), schedule_managers}
    end)
  end

  def update_stream_status(%{status: "streaming"} = manager),
    do:
      ScheduleManager.update!(manager, %{
        status: "recovering",
        updated_at: NaiveDateTime.utc_now(),
        last_started_at: NaiveDateTime.add(NaiveDateTime.utc_now(), 15)
      })

  def update_stream_status(manager),
    do:
      ScheduleManager.update!(manager, %{
        status: "streaming",
        updated_at: NaiveDateTime.utc_now(),
        last_started_at: NaiveDateTime.add(NaiveDateTime.utc_now(), 15)
      })

  def scheduled_to_play(%{status: status}) when status in ["streaming", "__FORCE_START__"],
    do: true

  def scheduled_to_play(%{cron: cron, last_started_at: last_started_at}) when is_nil(cron),
    do: false

  def scheduled_to_play(
        %{cron: cron, last_started_at: last_started_at, inserted_at: inserted_at} = airing
      )
      when is_nil(last_started_at),
      do: scheduled_to_play(%{airing | last_started_at: inserted_at})

  def scheduled_to_play(%{cron: cron, last_started_at: last_started_at}),
    do: Utils.cron_triggered?(cron, last_started_at)

  defp waiting_or_streaming_managers() do
    from(
      a in ScheduleManager,
      where:
        a.status in ["waiting", "__FORCE_START__"] or
          (a.updated_at < ago(1, "minute") and a.status == "streaming"),
      select: a
    )
  end

  defp get_managers_with_lock(ids, limit) do
    from(
      a in ScheduleManager,
      where: a.id in ^ids,
      select: a,
      limit: ^limit,
      lock: "FOR UPDATE SKIP LOCKED"
    )
  end
end
