defmodule Streamer do
  defstruct [:chat_bot, :scheduler, :streamer, :schedule_manager]
  use GenServer
  require Logger
  alias MissionControlEx.Web.{ScheduleManager, StreamSchedule, Event, EventChunk, Channel}

  defmodule(
    EndStream,
    do: defstruct([:recurring])
  )

  @reset_after_duration 46 * 60 * 60 * 1000

  def start_link(schedule_manager) do
    channel = MissionControlEx.Web.Channel.get_channel(schedule_manager.channel_id)
    Logger.info("Initializing channel", channel: channel.login)

    {:ok, pid} =
      GenServer.start_link(
        Streamer,
        {schedule_manager, channel},
        name:
          {
            :global,
            {:stream, Integer.to_string(schedule_manager.id)}
          }
      )
  end

  def init({schedule_manager, channel} = args) do
    chat_bot =
      case ChatBot.start_chat_bot(args) do
        {:ok, chat_bot} -> chat_bot
        {:error, reason} -> nil
      end

    parent = self()
    task = Task.async(fn -> start_stream(schedule_manager, chat_bot) end)
    {:ok, %{streamer: task, schedule_manager: schedule_manager, chat_bot: chat_bot}}
  end

  def handle_cast(
        :stop,
        %{chat_bot: chat_bot, streamer: streamer, schedule_manager: schedule_manager} = state
      ) do
    if chat_bot, do: GenServer.stop(chat_bot)
    Task.shutdown(streamer)
    ScheduleManager.update(schedule_manager.id, %{status: "stopped"})
    {:stop, :normal, state}
  end

  def handle_info({_, {:ok, :stream_done}}, state), do: {:stop, :normal, state}

  def start_stream(schedule_manager, chat_bot) do
    own_schedule_manager_til_death(self(), schedule_manager)
    start_streaming(schedule_manager, chat_bot)
  end

  def start_streaming(
        %{id: schedule_manager_id, channel: channel, status: status} = schedule_manager,
        chat_bot
      )
      when status in ["streaming", "recovering"] do
    schedule_manager
    |> ScheduleManager.persist_stream_launched()
    |> ScheduleManager.generate_http_chunk_stream()
    |> EventChunk.chunk_stream_to_event_stream()
    |> add_byte_stream_to_event_stream(self())
    |> suspend_on_suspends(self())
    |> add_current_duration_to_stream
    |> end_stream_after(@reset_after_duration)
    |> remove_current_duration
    |> process_and_persist_events
    |> Stream.map(&end_on_stream_done(&1, self()))
    |> send_chat_bot_update(chat_bot)
    |> stream_to_twitch(channel, schedule_manager_id, chat_bot)
  end

  def start_streaming(%{id: schedule_manager_id, channel: channel} = schedule_manager),
    do: {:error, :stream_done, nil}

  def own_schedule_manager_til_death(pid, schedule_manager) do
    Task.Supervisor.start_child(MissionControlEx.Web.OwnershipSupervisor, fn ->
      Utils.wait_for_finish(pid, fn ->
        MissionControlEx.Web.ScheduleManager.touch(schedule_manager)
      end)
    end)

    {:ok, pid}
  end

  def get_state(pid) do
    GenServer.call(pid, :get_state)
  end

  def handle_call(:get_state, _from, state) do
    {:reply, state, state}
  end

  def handle_info({_, {:error, :stream_done, _schedule}}, state), do: {:stop, :normal, state}

  def add_byte_stream_to_event_stream(event_stream, parent) do
    Stream.flat_map(event_stream, fn event -> [event | Event.to_video_bytes(event)] end)
    |> bytes_buffer(3)
  end

  # def apply_buffer_flush(stream, schedule_manager, parent) do
  def process_and_persist_events(stream) do
    stream
    |> Stream.each(&do_process_persist_event/1)
  end

  def do_process_persist_event(%Event{} = event), do: Event.api_call_to_persist(event)

  def do_process_persist_event(_), do: :noop

  defp persist(schedule_manager, stream_schedule, event) do
    StreamSchedule.persist_changes(stream_schedule)
    MissionControlEx.Web.Repo.insert!(event)
    ScheduleManager.update(schedule_manager, %{status: schedule_manager.status})
  end

  def send_chat_bot_update(stream, chat_bot),
    do: Stream.each(stream, &do_send_chat_bot_update(&1, chat_bot))

  def do_send_chat_bot_update(%Event{} = event, nil), do: IO.inspect("no chat bot running")

  def do_send_chat_bot_update(%Event{} = event, chat_bot),
    do: Process.send_after(chat_bot, :send_auto_messages, 0)

  def do_send_chat_bot_update(thing, _), do: thing

  defp end_on_stream_done(%Event{type: "end_stream", data: %{"recurring" => recurring}}, parent) do
    send(parent, %EndStream{recurring: recurring})
    {:halt, :ok}
  end

  defp end_on_stream_done(event_block, parent), do: event_block

  def stream_to_twitch(stream, channel, schedule_manager_id, chat_bot) do
    stream =
      stream
      |> Stream.transform(:ok, &to_byte_stream/2)
      |> Stream.each(&assure_status_streaming(&1, schedule_manager_id))
      |> Stream.concat()

    stream_key = Channel.get_stream_key(channel)

    opts = [in: stream]

    Porcelain.exec(
      "ffmpeg",
      [
        "-re",
        "-i",
        "-",
        "-c:v",
        "libx264",
        "-b:v",
        "5M",
        "-bufsize",
        "5M",
        "-minrate",
        "5M",
        "-maxrate",
        "5M",
        "-vsync",
        "cfr",
        "-r",
        "30",
        "-x264opts",
        "keyint=120:min-keyint=120:scenecut=-1",
        "-pix_fmt",
        "yuv420p",
        "-vf",
        "scale=1920:1080:force_original_aspect_ratio=decrease,pad=1920:1080:(ow-iw)/2:(oh-ih)/2",
        "-profile:v",
        "main",
        "-preset",
        "veryfast",
        "-c:a",
        "aac",
        "-b:a",
        "128k",
        "-f",
        "flv",
        "rtmp://live.twitch.tv/app/#{stream_key}"
      ],
      opts
    )

    case wait_for_stream_end() do
      :resume ->
        ScheduleManager.load(schedule_manager_id)
        |> start_streaming(chat_bot)

      %EndStream{recurring: true} ->
        ScheduleManager.update!(schedule_manager_id, %{status: "waiting"})
        {:ok, :stream_done}

      %EndStream{recurring: false} ->
        ScheduleManager.update!(schedule_manager_id, %{status: "done"})
        {:ok, :stream_done}
    end
  end

  def assure_status_streaming(_, schedule_manager_id) do
    ScheduleManager.update(schedule_manager_id, %{status: "streaming"})
  end

  def wait_for_stream_end() do
    receive do
      {:suspend, duration} ->
        :timer.sleep(duration)
        :resume

      %EndStream{} = end_stream_struct ->
        end_stream_struct
    after
      # Delay between stream ending and coming back to life
      1_000 ->
        :resume
    end
  end

  def suspend_on_suspends(stream, parent),
    do: Stream.transform(stream, :ok, &suspend_if_suspends(&1, &2, parent))

  def suspend_if_suspends(thing, {:suspend, duration}, parent) do
    IO.inspect(duration, label: "suspending for")
    send(parent, {:suspend, duration})
    {:halt, :ok}
  end

  def suspend_if_suspends(
        %{type: "suspend", data: %{"duration" => duration}} = event,
        :ok,
        _
      ) do
    {[event], {:suspend, duration}}
  end

  def suspend_if_suspends(thing, :ok, _), do: {[thing], :ok}

  def add_current_duration_to_stream(stream) do
    Stream.transform(stream, 0, &add_current_duration/2)
  end

  def add_current_duration(%Event{} = event, time),
    do: {[{event, time + Event.duration(event)}], time + Event.duration(event)}

  def add_current_duration(thing, time), do: {[{thing, time}], time}

  def end_stream_after(stream, time),
    do: Stream.transform(stream, :ok, end_stream_if_overtime(time))

  def end_stream_if_overtime(time) do
    fn
      {_, duration}, :ok when duration > time -> {:halt, :ok}
      thing, :ok -> {[thing], :ok}
    end
  end

  def remove_current_duration(stream), do: Stream.map(stream, &elem(&1, 0))

  def to_byte_stream({:async_bytes, %Task{} = task}, :ok) do
    bytes = Task.await(task, 100_000)
    {[bytes], :ok}
  end

  def to_byte_stream({:bytes, bytes}, :ok), do: {[bytes], :ok}

  def to_byte_stream({:func, callback_func}, :ok) do
    Task.start(fn -> callback_func.() end)
    {[], :ok}
  end

  def to_byte_stream({:halt, :ok}, :ok), do: {:halt, :ok}
  def to_byte_stream(_, :ok), do: {[], :ok}

  def bytes_buffer(stream, size) do
    Stream.concat(stream, [:done]) |> Stream.transform(_empty_buffer = [], transform_buffer(size))
  end

  defp transform_buffer(size) do
    fn
      :done, buffer ->
        {buffer, :done}

      thing, buffer ->
        buffer_length = Enum.count(buffer, &is_video_buffer/1)

        case {thing, buffer, buffer_length} do
          {thing, buffer, buffer_length} when buffer_length < size ->
            {[], buffer ++ [thing]}

          {thing, [h | t], buffer_length} ->
            {[h], t ++ [thing]}
        end
    end
  end

  defp is_video_buffer({:bytes, _}), do: true
  defp is_video_buffer({:async_bytes, _}), do: true
  defp is_video_buffer(_), do: false
end
