defmodule RevCaptions do
  @moduledoc """
  Uses https://www.rev.com/api/docs
  Upload Flow
    Check if order exists by client_ref
    POST asset to Rev as S3 signed url
    POST caption order request to Rev
  Download Flow
    with asset
    GET caption order by client_ref (s3 transcode path)
    Extract Order Number
    GET Order by Order Number
    Extract Captions Attachment ID
    GET Captions by Attachment ID (defaults to subrip format `.srt`)
    PUT Captions to AWS s3 to path (asset subtitles destination, not source_subtitles)
  """
  use HTTPoison.Base
  require Utils
  alias MissionControlEx.Web.{Asset}

  @client_key %{
    dev: "z8lKW15wuy0a4147yb7uNHvcOm0",
    staging: "z8lKW15wuy0a4147yb7uNHvcOm0",
    prod: "ggaGNRcKh4FU4pYO1V4JKQo-3P4"
  }
  @user_key %{
    dev: "AAANTswnDlMNfr01BMOkDr1B5VM=",
    staging: "AAANTswnDlMNfr01BMOkDr1B5VM=",
    prod: "VNgCAKIHblLRBC+4Za7WOPuRi2M="
  }
  @rev_url %{
    dev: "https://api-sandbox.rev.com",
    staging: "https://api-sandbox.rev.com",
    prod: "https://www.rev.com"
  }
  @rev_version 1

  @rev_headers [
    Authorization: "Rev #{@client_key[Mix.env()]}:#{@user_key[Mix.env()]}",
    "Content-Type": "application/json",
    Accept: "application/json"
  ]

  @httpoison_options [timeout: 60_000, recv_timeout: 60_000]
  @exaws_config ExAws.Config.new(:s3, %{
                  access_key_id: "AKIAJ4RHGC5TBNDDSR3A",
                  secret_access_key: "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"
                })

  @doc """
  The client_ref given to Rev is always the asset's transcoded s3 path (s3 transcode path = "prod/mst3k/episode99/transcode/episode99.flv")
  """
  def process_url(url), do: @rev_url[Mix.env()] <> "/api/v#{@rev_version}" <> url

  @doc """
  Entry for ordering captions for an asset
  """
  def check_order_asset(asset) do
    order = get_order_by_client_ref(asset)

    case order do
      nil -> add_input_to_rev(asset)
      order -> nil
    end
  end

  def order_captions(asset, location) do
    req_body =
      %{
        client_ref: asset_client_ref(asset),
        caption_options: %{
          inputs: [
            %{
              video_length_seconds: asset.raw_duration,
              uri: location
            }
          ]
        }
      }
      |> Poison.encode!()

    %{status_code: 201} = post!("/orders", req_body, @rev_headers, @httpoison_options)
  end

  def add_input_to_rev(asset) do
    # presigned url expires in 1 weeks
    # NOTE: providing transcoded asset (.flv)
    {:ok, public_url} =
      ExAws.S3.presigned_url(
        @exaws_config,
        :get,
        "twitch-creative-video-repository",
        Transcode.s3_path(:transcode, asset),
        expires_in: 7 * 24 * 60 * 60
      )

    filename = Path.basename(Transcode.s3_path(:transcode, asset))

    req_body =
      %{
        "content_type" => "video/x-flv",
        "filename" => filename,
        "url" => public_url
      }
      |> Poison.encode!()

    {:ok, %HTTPoison.Response{}} =
      post(
        "/inputs",
        req_body,
        @rev_headers,
        timeout: 60_000_000,
        recv_timeout: 60_000_000
      )
  end

  @doc """
  Primary Entry for getting Captions
  """
  def get_captions(asset) do
    get_order_by_client_ref(asset)
    |> get_order_by_order
    |> get_attachments_of_order
    |> download_attachment
    |> upload_captions_to_s3(asset)
  end

  def get_order_by_client_ref(asset) do
    %{status_code: 200, body: resp_body} =
      get!("/orders?clientRef=#{asset_client_ref(asset)}", @rev_headers, @httpoison_options)

    resp_body
    |> Poison.decode!()
    |> Map.get("orders")
    |> Enum.find(fn order -> order["client_ref"] == asset_client_ref(asset) end)
  end

  def get_order_by_order(%{"status" => "Complete", "order_number" => order_number} = order) do
    %{status_code: 200, body: resp_body} =
      get!("/orders/#{order_number}", @rev_headers, @httpoison_options)

    resp_body
    |> Poison.decode!()
  end

  def get_order_by_order(%{"status" => _status} = order), do: nil

  def get_attachments_of_order(nil), do: nil
  def get_attachments_of_order(nil, _type), do: nil

  def get_attachments_of_order(order, type \\ "caption") do
    order
    |> Map.get("attachments")
    |> Enum.find(fn attachment -> attachment["kind"] == type end)
  end

  def download_attachment(nil), do: nil

  def download_attachment(
        %{"id" => id, "name" => name, "links" => links} = attachment,
        extname \\ ".srt"
      ) do
    headers = @rev_headers ++ [Accept: "application/x-subrip"]
    {:ok, caption_path} = Briefly.create(extname: extname)
    %{status_code: 200, body: captions} = get!("/attachments/#{id}/content", headers)
    :ok = File.write!(caption_path, captions)
    caption_path
  end

  def upload_captions_to_s3(nil, _), do: nil

  def upload_captions_to_s3(caption_path, asset) do
    %{status_code: 200} =
      ExAws.S3.put_object(
        "twitch-creative-video-repository",
        Transcode.s3_path(:subtitles, asset),
        File.read!(caption_path)
      )
      |> ExAws.request!(@exaws_config)
  end

  def asset_client_ref(asset),
    do:
      Transcode.s3_path(:transcode, asset)
      |> String.replace(["'", "\"", " ", "+"], "")

  @doc """
  Util functions
  """
  def get_status_of_order(asset) do
    case get_order_by_client_ref(asset) do
      nil -> nil
      order -> Map.get(order, "status", nil)
    end
  end

  def get_orders() do
    %{status_code: 200, body: body} =
      get!("/orders?pageSize=100", @rev_headers, @httpoison_options)

    Poison.decode!(body)
  end

  def incomplete_orders(results, total, _page) when total < 0, do: results

  def incomplete_orders(results, total, page) do
    %{status_code: 200, body: body} =
      get!("/orders?pageSize=100&page=#{page}", @rev_headers, @httpoison_options)

    new_results =
      Poison.decode!(body)
      |> Map.get("orders")
      |> Enum.filter(fn order -> order["status"] != "Complete" end)

    incomplete_orders(results ++ new_results, total - 100, page + 1)
  end

  def incomplete_orders() do
    %{status_code: 200, body: body} =
      get!("/orders?pageSize=100", @rev_headers, @httpoison_options)

    body = Poison.decode!(body)
    total = body |> Map.get("total_count")

    results =
      body |> Map.get("orders") |> Enum.filter(fn order -> order["status"] != "Complete" end)

    incomplete_orders(results, total - 100, 0 + 1)
  end

  def cancel_order(nil), do: nil

  def cancel_order(%Asset{} = asset) do
    get_order_by_client_ref(asset)
    |> cancel_order
  end

  def cancel_order(%{"order_number" => order_number}), do: cancel_order(order_number)

  def cancel_order(order_number) when is_bitstring(order_number) do
    post!("/orders/#{order_number}/cancel", "", @rev_headers, @httpoison_options)
  end
end
