defmodule MissionControlEx.Web.TranscoderJobProducer do
  use GenStage
  alias MissionControlEx.Web.Repo
  import Ecto.Query
  @five_seconds 5_000
  def start_link do
    Task.async(&poll_queue/0)
    GenStage.start_link(__MODULE__, 0, name: __MODULE__)
  end

  def poll_queue do
    :timer.sleep(@five_seconds)
    send(MissionControlEx.Web.TranscoderJobProducer, :transcode_job_enqueued)
    poll_queue()
  end

  def init(counter) do
    {:producer, counter}
  end

  def handle_demand(demand, state) do
    limit = demand + state
    {:ok, {count, events}} = take(limit)
    {:noreply, events, limit - count}
  end

  def handle_info(:transcode_job_enqueued, state) do
    limit = state
    {:ok, {count, events}} = take(limit)
    {:noreply, events, limit - count}
  end

  def take(limit) do
    Repo.transaction(fn ->
      ids = Repo.all(waiting_assets(limit))

      {count, ids} =
        Repo.update_all(
          assets_by_ids(ids),
          [set: [status: "transcoding", updated_at: Ecto.DateTime.utc()]],
          returning: [:id]
        )

      id_list = ids |> Enum.flat_map(&Map.values/1)

      assets = Repo.all(from(a in MissionControlEx.Web.Asset, where: a.id in ^id_list))

      {count, assets}
    end)
  end

  defp assets_by_ids(ids) do
    from(a in "assets", where: a.id in ^ids)
  end

  defp waiting_assets(demand) do
    from(
      a in "assets",
      where:
        a.status == "waiting" or
          (a.updated_at < ago(2, "minute") and a.status not in ["complete", "waiting_on_captions"] and
             not ilike(a.status, "error%")) or
          (a.updated_at < ago(6, "hour") and a.status == "waiting_on_captions") or
          (a.updated_at < ago(1, "week") and ilike(a.status, "error%")),
      limit: ^demand,
      select: a.id,
      lock: "FOR UPDATE SKIP LOCKED"
    )
  end
end
