defmodule Utils do
  @max_runs 100_000_000
  def get_type(".mov"), do: :video
  def get_type(".mp4"), do: :video
  def get_type(".avi"), do: :video
  def get_type(".wmv"), do: :video
  def get_type(".mkv"), do: :video
  def get_type(".flv"), do: :video
  def get_type(".mpg"), do: :video
  def get_type(".MPG"), do: :video
  def get_type(".ts"), do: :video
  def get_type(".scc"), do: :subtitles
  def get_type(".srt"), do: :subtitles
  def get_type(".dfxp"), do: :subtitles
  def get_type(".sami"), do: :subtitles
  def get_type(".ts?" <> _rest), do: :video
  def get_type(".mpg?" <> _rest), do: :video
  def get_type(".MPG?" <> _rest), do: :video
  def get_type(".mov?" <> _rest), do: :video
  def get_type(".mp4?" <> _rest), do: :video
  def get_type(".avi?" <> _rest), do: :video
  def get_type(".wmv?" <> _rest), do: :video
  def get_type(".mkv?" <> _rest), do: :video
  def get_type(".flv?" <> _rest), do: :video
  def get_type(".scc?" <> _rest), do: :subtitles
  def get_type(".srt?" <> _rest), do: :subtitles
  def get_type(".dfxp?" <> _rest), do: :subtitles
  def get_type(".sami?" <> _rest), do: :subtitles
  def get_type(<<_first_letter::size(8)>> <> rest), do: get_type(rest)
  def get_type(<<>>), do: :no_type

  def get_extname(".mov"), do: ".mov"
  def get_extname(".mp4"), do: ".mp4"
  def get_extname(".avi"), do: ".avi"
  def get_extname(".wmv"), do: ".wmv"
  def get_extname(".mkv"), do: ".mkv"
  def get_extname(".flv"), do: ".flv"
  def get_extname(".scc"), do: ".scc"
  def get_extname(".srt"), do: ".srt"
  def get_extname(".dfxp"), do: ".dfxp"
  def get_extname(".sami"), do: ".sami"
  def get_extname(".mpg"), do: ".mpg"
  def get_extname(".MPG"), do: ".MPG"
  def get_extname(".ts"), do: ".ts"
  def get_extname(".ts?" <> _rest), do: ".ts"
  def get_extname(".mpg?" <> _rest), do: ".mpg"
  def get_extname(".MPG?" <> _rest), do: ".MPG"
  def get_extname(".mov?" <> _rest), do: ".mov"
  def get_extname(".mp4?" <> _rest), do: ".mp4"
  def get_extname(".avi?" <> _rest), do: ".avi"
  def get_extname(".wmv?" <> _rest), do: ".wmv"
  def get_extname(".mkv?" <> _rest), do: ".mkv"
  def get_extname(".flv?" <> _rest), do: ".flv"
  def get_extname(".scc?" <> _rest), do: ".scc"
  def get_extname(".srt?" <> _rest), do: ".srt"
  def get_extname(".dfxp?" <> _rest), do: ".dfxp"
  def get_extname(".sami?" <> _rest), do: ".sami"
  def get_extname(<<_first_letter::size(8)>> <> rest), do: get_extname(rest)
  def get_extname(<<>>), do: :no_extname

  def get_top_level(), do: get_top_level(Mix.env())
  def get_top_level(:dev), do: "dev/"
  def get_top_level(:staging), do: "staging/"
  def get_top_level(:stagingW), do: "staging/"
  def get_top_level(:prod), do: "prod/"
  def get_top_level(:prodW), do: "prod/"
  def get_top_level(:prodX), do: "prodX/"
  def get_top_level(_env), do: "dev/"

  def content_type(".flv"), do: "video/x-flv"
  def content_type(".ts"), do: "video/vnd.dlna.mpeg-tts"
  def content_type(".srt"), do: "text/plain"
  def content_type(_), do: "application/octet-stream"

  # s3_path example: staging_final/clips/COSMOS/DependableRelievedNightingaleShazBotstix-DeepAstronomy.flv
  def get_channel_name_from_asset(s3_path) do
    # channel name
    s3_path
    |> String.split("/")
    |> Enum.at(2)
    |> String.downcase()
  end

  def wait_for_finish(pid, touch_func, finish_func \\ fn -> nil end) do
    receive do
    after
      10_000 ->
        if Process.alive?(pid) do
          touch_func.()
          wait_for_finish(pid, touch_func, finish_func)
        else
          finish_func.()
        end
    end
  end

  def rollbar_report_warning(error, %{} = metadata) do
    {mgsec, sec, _usec} = :os.timestamp()
    rollbar_body = Rollbax.Item.message_to_body(error, metadata)
    Rollbax.Client.emit(:warn, mgsec * 1_000_000 + sec, rollbar_body, %{}, %{})
  end

  def random_string(length) do
    :crypto.strong_rand_bytes(length) |> Base.url_encode64() |> binary_part(0, length)
  end

  def cron_triggered?(cron, timestamp) do
    {:ok, next_scheduled_time} = Crontab.Scheduler.get_next_run_date(cron, timestamp, @max_runs)
    # Running 3 seconds ahead to account for ffmpeg spinup time
    trigger_time =
      NaiveDateTime.utc_now()
      |> NaiveDateTime.add(3, :second)

    case NaiveDateTime.compare(trigger_time, next_scheduled_time) do
      :lt -> false
      :gt -> true
      :eq -> true
    end
  end

  def time_compare?(time1, time2) do
    case Timex.compare(time1, time2) do
      1 -> true
      0 -> true
      _ -> false
    end
  end

  def get_keys(list_of_maps, key \\ %{})
  def get_keys([], key), do: Map.keys(key)
  def get_keys([%{} = h | t], key), do: get_keys(t, Map.merge(key, h))
  def get_keys([_not_map | t], keys), do: get_keys(t, keys)

  # Convert metadata string key map to metadata atom key map, useful for EEx string interpolation
  def string_keys_to_atoms(map),
    do: for({key, val} <- map, into: %{}, do: {String.to_atom(key), val})

  # accepts hh:mm:ss:ms or hh:mm:ss;ms
  def timestamp_to_ms(ts) do
    [hh, mm, ss, ms] =
      String.split(ts, [":", ";"])
      |> Enum.map(&String.to_integer/1)

    hh * 3600 * 1000 + mm * 60 * 1000 + ss * 1000 + ms
  end

  def create_tmp_file(extname, timeout \\ 60_000) do
    opts = [extname: extname]
    GenServer.call(Briefly.Entry.server(), {:create, opts}, timeout)
  end

  def pid_to_string(pid), do: :erlang.pid_to_list(pid) |> to_string

  def parse_int(num) when is_integer(num), do: num

  def parse_int(num) do
    case Integer.parse(num) do
      {int, _} -> int
      err -> err
    end
  end

  def time_to_run(func) do
    func
    |> :timer.tc()
    |> elem(0)
    |> Kernel./(1_000_000)
  end
end

defimpl Poison.Encoder, for: Crontab.CronExpression do
  def encode(crontab, options) do
    Crontab.CronExpression.Composer.compose(crontab)
    |> Poison.Encoder.BitString.encode(options)
  end
end

defimpl Phoenix.HTML.Safe, for: Crontab.CronExpression do
  def to_iodata(data), do: Crontab.CronExpression.Composer.compose(data)
end

defimpl Poison.Encoder, for: [NaiveDateTime] do
  def encode(dt, _opts), do: <<?", @for.to_iso8601(dt)::binary, ?Z, ?">>
end
