defmodule MissionControlEx.Web.ChatBotTest do
  use MissionControlEx.Web.ModelCase
  import MissionControlEx.Web.Factory

  alias MissionControlEx.Web.Channel
  alias MissionControlEx.Web.Asset
  alias MissionControlEx.Web.StreamSchedule

  def basic_loop(asset1, asset2),
    do:
      Poison.decode!("""
      {
      "events": [
        {
          "type": "loop",
          "ref": "main_loop",
          "events": [
            {
              "type": "asset",
              "id": #{asset1.id},
              "ref": "episode1"
            },
            {
              "type": "asset",
              "id": #{asset2.id},
              "ref": "episode2"
            }
          ]
        }
      ]}
      """)

  def basic_chat_seed() do
    channel =
      insert(:channel, %{
        login: "tdevtest",
        oauth_token: "bs2kxliwimvkqmzezh8tphn00disjl",
        chat_bot_enabled: true
      })

    {:ok, cron} = Crontab.CronExpression.Parser.parse("* * * * * *", true)

    manager =
      insert(:schedule_manager, %{
        channel: channel,
        cron: cron,
        status: "waiting"
      })

    asset1 =
      insert(:asset, %{
        s3_path: "test/colorful_timer_5min.ts",
        duration: 300_000,
        type: "episode",
        commercial_breaks_ms: [30_000],
        metadata: %{episode: "1", season: "1", title: "This is a title"},
        status: "complete"
      })

    asset2 =
      insert(:asset, %{
        s3_path: "test/2min.ts",
        duration: 60_000,
        metadata: %{episode: "2", season: "1"},
        commercial_breaks_ms: [],
        status: "complete"
      })

    twitch_stream =
      insert(:stream_schedule, %{
        state: "waiting",
        name: "first_stream",
        default_schedule: true,
        schedule_manager_id: manager.id,
        stack_trace: ["episode1", "main_loop"],
        stream_schema: basic_loop(asset1, asset2)
      })

    chat_command =
      insert(:chat_command, %{
        command: "!current",
        message_template:
          "Current episode <%= data[\"current\"][\"episode\"] %> of season <%= data[\"current\"][\"season\"] %>",
        fallback_message: "Can't tell you what's playing, not sure why"
      })

    chat_command2 =
      insert(:chat_command, %{
        command: "!next",
        message_template:
          "Next episode <%= data[\"next\"][\"episode\"] %> of season <%= data[\"next\"][\"season\"] %>",
        fallback_message: "Next broke"
      })

    chat_command3 =
      insert(:chat_command, %{
        command: "!currentlybroke",
        message_template:
          "Broken <%= data[\"nope\"][\"episode\"] %> of season <%= data[\"nope\"][\"season\"] %>"
      })

    insert(:channel_command, %{
      channel: channel,
      chat_command: chat_command
    })

    insert(:channel_command, %{
      channel: channel,
      chat_command: chat_command2
    })

    insert(:channel_command, %{
      channel: channel,
      chat_command: chat_command3
    })

    Repo.insert!(MissionControlEx.Web.Event.do_loop(twitch_stream.id, "main_loop"))

    Repo.insert!(
      MissionControlEx.Web.Event.play_asset(twitch_stream.id, "episode1", asset1, 0, "")
    )

    Channel.load_with_chat_data(channel)
  end

  @tag :skip
  test "get chat data correctly" do
    channel = basic_chat_seed
    assert channel.chat_data["current"]["title"] == "This is a title"
    assert channel.chat_data["next"]["episode"] == "2"
  end

  @tag :skip
  test "Generate message for current and next correctly" do
    channel = basic_chat_seed

    message_1 =
      Enum.at(channel.chat_commands, 0)
      |> ChatBotMessager.generate_message(channel)

    assert message_1 == "Current episode 1 of season 1"

    message_2 =
      Enum.at(channel.chat_commands, 1)
      |> ChatBotMessager.generate_message(channel)

    assert message_2 == "Next episode 2 of season 1"
  end

  @tag :skip
  test "Fail to generate message if metadata isn't present" do
    channel = basic_chat_seed

    fail_message =
      Enum.at(channel.chat_commands, 2)
      |> ChatBotMessager.generate_message(channel)

    assert fail_message == {:error, :no_fallback_message}
  end
end
