defmodule EventControllerTest do
  use ExUnit.Case, async: true
  # use Plug.Test
  use MissionControlEx.Web.ConnCase
  use MissionControlEx.Web.ModelCase

  import MissionControlEx.Web.Factory
  import MissionControlEx.Test.TestHelpers
  alias MissionControlEx.Web.{Router, Repo, Event, StreamSchedule, ScheduleManager, EventChunk}

  def do_setup(csv_schedule_string \\ nil) do
    insert_assets_with_commercial_breaks(10)
    {:ok, cron} = Crontab.CronExpression.Parser.parse("* * * * * *", true)

    channel =
      insert(:channel, %{
        login: "TEST_CHANNEL",
        oauth_token: "invalid token"
      })

    schedule_manager =
      insert(:schedule_manager, %{cron: cron, status: "waiting", channel: channel})
      |> ScheduleManager.load()

    csv_schedule_string =
      csv_schedule_string ||
        """
        schedule_block_length,asset_path,commercial_path
        0:02:00,1.ts,fallback.ts
        0:02:00,2.ts,fallback.ts
        """

    stream_schedule =
      csv_schedule_string
      |> csv_string_to_stream
      |> make_schedule_with_csv(%{
           schedule_manager_id: schedule_manager.id,
           repeat: false
         })

    {stream_schedule, schedule_manager}
  end

  test "persist event and update stream schedule", %{conn: conn} do
    {stream_schedule, schedule_manager} = do_setup()
    start_time = ~N[2017-04-20 16:20:00.0000]

    %{event: target_event} =
      schedule_manager
      |> ScheduleManager.load()
      |> ScheduleManager.generate_event_stream(start_time)
      |> Stream.take(1)
      |> Enum.at(-1)

    req_body = %{event: Poison.encode!(target_event)}

    path = event_path(conn, :process_and_persist_event)

    conn =
      conn
      |> put_req_header("content-type", "application/json")
      |> post(path, req_body)

    total_events = length(stream_schedule.queued_event_chunks)
    assert schedule_manager.events == []

    assert conn.state == :sent
    assert conn.status == 204

    %{events: events, stream_schedules: [stream_schedule]} =
      ScheduleManager.load(schedule_manager.id)

    assert length(events) == 1
    [persisted_event] = events
    assert %{data: %{"asset" => %{"s3_path" => "1.ts"}}} = persisted_event

    assert Map.take(target_event, [:data, :chunk_id, :chunk_index]) ==
             Map.take(persisted_event, [:data, :chunk_id, :chunk_index])

    %{played_event_chunks: played, queued_event_chunks: chunk_list} = stream_schedule

    assert length(played) == 1
    assert length(chunk_list) == total_events - 1
    assert stream_schedule.chunk_index == 0
  end

  test "process event, sending as binary", %{conn: conn} do
    {stream_schedule, schedule_manager} = do_setup()
    path = event_path(conn, :process_and_persist_event)
    start_time = ~N[2017-04-20 16:20:00.0000]

    %{event: asset_event, schedule_manager: schedule_manager} =
      schedule_manager
      |> ScheduleManager.load()
      |> ScheduleManager.generate_event_stream(start_time)
      |> Stream.take(1)
      |> Enum.at(-1)

    assert %{data: %{"asset" => %{"s3_path" => "1.ts"}}, chunk_index: 0} = asset_event
    req_body = %{event: Poison.encode!(asset_event)}

    conn =
      conn
      |> put_req_header("content-type", "application/json")
      |> post(path, req_body)

    assert conn.state == :sent
    assert conn.status == 204

    %{events: events, stream_schedules: [stream_schedule]} =
      ScheduleManager.load(schedule_manager.id)

    assert length(events) == 1
    assert [%{data: %{"asset" => %{"s3_path" => "1.ts"}}}] = events

    %{event: target_event, stream_schedule: stream_schedule, schedule_manager: manager} =
      schedule_manager
      |> ScheduleManager.load()
      |> Map.put(:status, "streaming")
      |> ScheduleManager.generate_event_stream()
      |> Stream.take(1)
      |> Enum.at(-1)

    assert %{data: %{"asset" => %{"s3_path" => "fallback.ts"}}, chunk_index: 1} = target_event
    req_body = %{event: Poison.encode!(target_event)}

    conn =
      build_conn()
      |> put_req_header("content-type", "application/json")
      |> post(path, req_body)

    assert conn.state == :sent
    assert conn.status == 204

    %{events: events, stream_schedules: [stream_schedule]} =
      ScheduleManager.load(schedule_manager.id)

    assert length(events) == 2

    assert [
             %{data: %{"asset" => %{"s3_path" => "1.ts"}}},
             %{data: %{"asset" => %{"s3_path" => "fallback.ts"}}}
           ] = events

    assert length(stream_schedule.played_event_chunks) == 1
    assert stream_schedule.chunk_index == 1
  end

  test "process event, sending as stringified JSON", %{conn: conn} do
    {stream_schedule, schedule_manager} = do_setup()
    path = event_path(conn, :process_and_persist_event)
    start_time = ~N[2017-04-20 16:20:00.0000]

    %{event: asset_event, schedule_manager: schedule_manager} =
      schedule_manager
      |> ScheduleManager.load()
      |> ScheduleManager.generate_event_stream(start_time)
      |> Stream.take(1)
      |> Enum.at(-1)

    assert %{data: %{"asset" => %{"s3_path" => "1.ts"}}, chunk_index: 0} = asset_event
    req_body = %{event: Poison.encode!(asset_event)}

    conn =
      conn
      |> put_req_header("content-type", "application/json")
      |> post(path, req_body)

    assert conn.state == :sent
    assert conn.status == 204

    %{events: events, stream_schedules: [stream_schedule]} =
      ScheduleManager.load(schedule_manager.id)

    assert length(events) == 1
    assert [%{data: %{"asset" => %{"s3_path" => "1.ts"}}}] = events
  end
end
