defmodule MissionControlEx.Web.AssetTest do
  use MissionControlEx.Web.ModelCase

  import MissionControlEx.Web.Factory

  alias MissionControlEx.Web.{Repo, Asset}

  test "pulls assets by metafield" do
    episodes = [
      insert(:asset, %{
        type: "episode",
        s3_path: "3.flv",
        status: "complete",
        metadata: %{"great_episode_seq" => 1}
      }),
      insert(:asset, %{
        type: "episode",
        s3_path: "2.flv",
        status: "complete",
        metadata: %{"great_episode_seq" => 0}
      }),
      insert(:asset, %{type: "episode", s3_path: "1.flv", status: "complete"}),
      insert(:asset, %{type: "episode", s3_path: "4.flv", status: "complete"})
    ]

    assets = Asset.get_assets_by_metafield("great_episode_seq")
    assert length(assets) == 2
    assert Enum.sort_by(assets, & &1.metadata["great_episode_seq"]) == assets
  end

  test "delete metafield from assets" do
    asset_1 =
      insert(:asset, %{
        type: "episode",
        source_movie: "1.mp4",
        s3_path: "3.flv",
        status: "complete",
        metadata: %{"great_episode_seq" => 1, "no_deleterino" => true}
      })

    asset_2 =
      insert(:asset, %{
        type: "episode",
        source_movie: "2.mp4",
        s3_path: "2.flv",
        status: "complete",
        metadata: %{"great_episode_seq" => 0}
      })

    assets = Asset.get_assets_by_metafield("great_episode_seq")
    assert length(assets) == 2

    Asset.update_metafields([
      {asset_1, %{"great_episode_seq" => "__delete__"}},
      {asset_2, %{"great_episode_seq" => "__delete__"}}
    ])

    assets = Asset.get_assets_by_metafield("great_episode_seq")
    assert length(assets) == 0
    assets = Asset.get_assets_by_metafield("no_deleterino")
    assert length(assets) == 1
  end

  test "update metafield for assets" do
    asset_1 =
      insert(:asset, %{
        source_movie: "1.mp4",
        type: "episode",
        s3_path: "3.flv",
        status: "complete",
        metadata: %{"great_episode_seq" => "1"}
      })

    asset_2 =
      insert(:asset, %{
        source_movie: "2.mp4",
        type: "episode",
        s3_path: "2.flv",
        status: "complete",
        metadata: %{"great_episode_seq" => 0}
      })

    asset_3 =
      insert(:asset, %{
        source_movie: "3.mp4",
        type: "episode",
        s3_path: "1.flv",
        status: "complete"
      })

    Asset.update_metafields([
      {asset_3, %{"great_episode_seq" => "2"}},
      {asset_1, %{"great_episode_seq" => "11"}}
    ])

    assets = Asset.get_assets_by_metafield("great_episode_seq")
    assert length(assets) == 3

    assert Enum.sort_by(assets, & &1.metadata["great_episode_seq"]) |> Enum.map(& &1.id) ==
             [asset_2, asset_3, asset_1] |> Enum.map(& &1.id)
  end
end
