defmodule MissionControlEx.Web.ManifestTest do
  use MissionControlEx.Web.ModelCase

  import MissionControlEx.Web.Factory

  alias MissionControlEx.Web.{Repo, Asset, Manifest, ManifestController}

  def insert_manifest(manifest_str, opts \\ %{}) do
    insert(:manifest, Map.merge(%{manifest: manifest_str}, opts))
  end

  test "does not generate multiple assets for same source_movie and trimmings" do
    manifest =
      """
      source_movie,trimmings
      test/blah.mov,""
      test/blah.mov,""
      """
      |> insert_manifest

    {:ok, assets} = Manifest.create_transcode_jobs(manifest)
    assert length(assets) == 1
  end

  test "multiple assets for different source_movie and trimmings" do
    manifest =
      """
      source_movie,trimmings
      test/blah.mov,"00:00:00,end"
      test/blah.mov,"11:11:11,end"
      test/blah.mov,"22:22:22,end"
      test/blah3.mov,"33:33:33,end"
      """
      |> insert_manifest

    {:ok, assets} = Manifest.create_transcode_jobs(manifest)
    assert length(assets) == 4
  end

  test "two manifests reuse a generic asset" do
    generic_asset = """
    source_movie,trimmings
    test/generic_asset.mov,"00:00:00,end"
    """

    manifest_1 = insert_manifest(generic_asset, %{name: "manifest_1"})
    manifest_2 = insert_manifest(generic_asset, %{name: "manifest_2"})
    {:ok, [asset_1]} = Manifest.create_transcode_jobs(manifest_1)
    {:ok, [asset_2]} = Manifest.create_transcode_jobs(manifest_2)
    assert asset_1.id == asset_2.id
    assert length(asset_2.manifests) == 2
  end

  test "two manifests reuse a generic asset with different trimmings" do
    manifest_1 =
      """
      source_movie,trimmings
      test/generic_asset.mov,""
      test/generic_asset.mov,"11:11:11,end"
      """
      |> insert_manifest(%{name: "manifest_1"})

    manifest_2 =
      """
      source_movie,trimmings
      test/generic_asset.mov,""
      test/generic_asset.mov,"00:22:22,end"
      test/another_asset.mov,"11:11:11,end"
      """
      |> insert_manifest(%{name: "manifest_2"})

    Manifest.create_transcode_jobs(manifest_1)
    Manifest.create_transcode_jobs(manifest_2)

    assets = Repo.all(Asset) |> Repo.preload(:manifests)
    assert length(assets) == 4

    asset_no_trim = Enum.find(assets, &(&1.trimmings == ""))

    asset_generic_1111 =
      Enum.find(
        assets,
        &(&1.trimmings == "11:11:11,end" && &1.source_movie == "test/generic_asset.mov")
      )

    asset_generic_2222 = Enum.find(assets, &(&1.trimmings == "00:22:22,end"))
    asset_another = Enum.find(assets, &(&1.source_movie == "test/another_asset.mov"))

    assert length(asset_no_trim.manifests) == 2
    assert Enum.all?(asset_no_trim.manifests, &(&1.id in [manifest_1.id, manifest_2.id]))
    assert length(asset_generic_1111.manifests) == 1
    assert length(asset_generic_2222.manifests) == 1
    assert length(asset_another.manifests) == 1
    assert List.first(asset_generic_1111.manifests).id == manifest_1.id
    assert List.first(asset_generic_2222.manifests).id == manifest_2.id
  end

  test "new manifest creates new twitch_stream of same manifest name with associated stream_assets" do
    name = "test_stream"

    manifest =
      """
      source_movie,trimmings
      test/1.mov,"11:11:11,end"
      test/2.mov,"22:22:22,end"
      """
      |> insert_manifest(%{name: name})

    {:ok, assets} = Manifest.create_transcode_jobs(manifest)
    twitch_stream = Manifest.create_twitch_stream(assets, manifest)

    assert length(assets) == 2
    assert twitch_stream.name == name
  end

  test "key assets fields not nil for manifest ingest" do
    name = "test_stream"

    manifest =
      """
      source_movie,trimmings,commercial_breaks_ms
      test/1.mov,"11:11:11,end",
      """
      |> insert_manifest(%{name: name})

    {:ok, [asset]} = Manifest.create_transcode_jobs(manifest)
    not_null_fields = [:trimmings, :commercial_breaks_ms]
    assert !Enum.any?(not_null_fields, fn field -> Map.get(asset, field) == nil end)
  end

  test "puts unknown csv columns into metadata at manifest ingest" do
    manifest =
      """
      source_movie,DANGER,SQL Attack,commercial_breaks_ms,episode,season,title
      test/1.mov,Yes?,;DROP TABLE assets;,[],99,1,Before the turn of a century
      """
      |> insert_manifest(%{name: "test_stream"})

    {:ok, [asset]} = Manifest.create_transcode_jobs(manifest)

    not_null_fields = [:trimmings, :commercial_breaks_ms]
    assert !Enum.any?(not_null_fields, fn field -> Map.get(asset, field) == nil end)
    assert asset.commercial_breaks_ms == []

    expected_metadata = %{
      "DANGER" => "Yes?",
      "SQL Attack" => ";DROP TABLE assets;",
      "episode" => "99",
      "season" => "1",
      "title" => "Before the turn of a century",
      "manifest_id" => manifest.id,
      "manifest_name" => manifest.name,
      "manifest_order_seq" => 0
    }

    assert asset.metadata == expected_metadata
  end

  test "updates manifest with metadata and removes old assocs" do
    manifest =
      """
      source_movie,title
      test/1.mov,"BLANK"
      test/2.mov,
      """
      |> insert_manifest(%{name: "test_stream"})

    {:ok, old_assets} = Manifest.create_transcode_jobs(manifest)
    old_assets = Repo.preload(old_assets, :manifests)

    assert length(old_assets) == 2

    assert Enum.all?(old_assets, fn asset ->
             [assoc] = asset.manifests
             assoc == manifest
           end)

    changes = %{
      "manifest" => """
      source_movie,title
      test/1.mov,"TEST 1"
      test/3.mov,"TEST 3"
      """
    }

    changeset = Manifest.changeset(manifest, changes)

    new_manifest =
      Repo.update!(changeset)
      |> Repo.preload(:assets)

    {:ok, updated_assets} = Manifest.create_transcode_jobs(new_manifest)

    asset1 = Enum.find(updated_assets, &(&1.source_movie == "test/1.mov"))
    asset3 = Enum.find(updated_assets, &(&1.source_movie == "test/3.mov"))

    assert asset1.metadata == %{
             "title" => "TEST 1",
             "manifest_id" => manifest.id,
             "manifest_name" => manifest.name,
             "manifest_order_seq" => 0
           }

    assert asset3.metadata == %{
             "title" => "TEST 3",
             "manifest_id" => manifest.id,
             "manifest_name" => manifest.name,
             "manifest_order_seq" => 1
           }

    [a1_manifest] = asset1.manifests
    [a3_manifest] = asset3.manifests
    assert a1_manifest.id == new_manifest.id
    assert a3_manifest.id == new_manifest.id

    asset2 =
      Repo.get_by(MissionControlEx.Web.Asset, source_movie: "test/2.mov")
      |> Repo.preload(:manifests)

    assert asset2.manifests == []
    assert asset2.metadata == %{}
  end
end
