defmodule MissionControlEx.Web.ScheduleCreationTest do
  use MissionControlEx.Web.ModelCase

  import MissionControlEx.Web.Factory
  import MissionControlEx.Test.TestHelpers
  alias MissionControlEx.Web.{Asset, Event, Schedule, StreamSchedule, ScheduleManager}
  @empty []

  @test_csv_list File.stream!("test/schema/test_schedule.csv")
                 |> CSV.decode!(headers: true)
                 |> Enum.to_list()

  @test_csv File.stream!("test/schema/test_schedule.csv")

  test "decodes into schema empty" do
    assert Schedule.to_event_list(@empty) == []
  end

  test "decodes into events with padded commercials" do
    insert_assets(8)
    event_list = Schedule.to_event_list(@test_csv_list)
    # Enum.each(event_list, &Event.do_print_events/1)
    assert length(event_list) == 7
    assert length(Enum.at(event_list, 0).events) == 1
    assert length(Enum.at(event_list, 1).events) == 1

    %{events: asset_3_events} = Enum.at(event_list, 3)
    assert length(asset_3_events) == 2
    assert Enum.at(asset_3_events, 0).type == "play_asset"
    assert Enum.at(asset_3_events, 1).type == "play_commercial"
  end

  test "decodes into schema with padded commercials and commercial breaks" do
    insert_assets_with_commercial_breaks(8)
    event_list = Schedule.to_event_list(@test_csv_list)
    assert length(event_list) == 7
    assert length(Enum.at(event_list, 0).events) == 3
    assert length(Enum.at(event_list, 1).events) == 3

    %{events: asset_3_events} = Enum.at(event_list, 3)
    assert length(asset_3_events) == 6
    assert Enum.at(asset_3_events, 4).type == "play_asset"
  end

  test "fills stream_schedule with correct events" do
    insert_assets(8)
    stream_schedule = make_schedule_with_csv(@test_csv)

    assert length(stream_schedule.queued_event_chunks) == 7

    {stream_schedule, %{events: asset_1_events}} =
      StreamSchedule.get_next_event_chunk(stream_schedule)

    assert length(asset_1_events) == 1
    assert Enum.at(asset_1_events, 0).data["asset"]["s3_path"] == "1.ts"

    [%{events: asset_3_events} | _] =
      stream_schedule
      |> ScheduleWorker.pop_queued_event_chunks()
      |> ScheduleWorker.pop_queued_event_chunks()
      |> Map.get(:played_event_chunks)

    assert length(asset_3_events) == 2
    assert Enum.at(asset_3_events, 0).data["asset"]["s3_path"] == "3.ts"
  end

  test "correctly allocates commercials" do
    insert_assets_with_commercial_breaks(5)
    # each asset is a minute long
    # each ad break gets padded with 4 seconds
    csv_stream =
      """
      schedule_block_length,asset_path,commercial_path
      0:02:00,1.ts,fallback.ts
      0:19:00,2.ts,fallback.ts
      0:00:30,3.ts,fallback.ts
      0:04:00,4.ts,fallback.ts
      ,5.ts,fallback.ts
      """
      |> csv_string_to_stream

    csv_list =
      csv_stream
      |> CSV.decode!(headers: true)
      |> Enum.to_list()

    event_list = Schedule.to_event_list(csv_list)
    assert length(event_list) == 5

    # evenly spread, mid rolls first
    event_chunk = Enum.at(event_list, 0)
    assert length(Enum.at(event_list, 0).events) == 5

    assert [
             %{type: "play_asset"},
             %{type: "play_commercial", data: %{"duration" => 34_000}},
             %{type: "play_asset"},
             %{type: "play_commercial", data: %{"duration" => 34_000}},
             %{type: "play_asset"}
           ] = event_chunk.events

    # allocates maximum
    event_chunk = Enum.at(event_list, 1)
    assert length(event_chunk.events) == 6

    assert [
             %{type: "play_asset"},
             %{type: "play_commercial", data: %{"duration" => 364_000}},
             %{type: "play_asset"},
             %{type: "play_commercial", data: %{"duration" => 364_000}},
             %{type: "play_asset"},
             %{type: "play_commercial", data: %{"duration" => 364_000}}
           ] = event_chunk.events

    # does not pad with ads when duration is shorter
    event_chunk = Enum.at(event_list, 2)
    assert length(event_chunk.events) == 3

    assert [
             %{type: "play_asset"},
             %{type: "play_asset"},
             %{type: "play_asset"}
           ] = event_chunk.events

    # evenly spreads ad duration, remainder front-loaded
    event_chunk_4 = Enum.at(event_list, 3)
    assert length(event_chunk_4.events) == 5

    assert [
             %{type: "play_asset"},
             %{type: "play_commercial", data: %{"duration" => 34_000}},
             %{type: "play_asset"},
             %{type: "play_commercial", data: %{"duration" => 34_000}},
             %{type: "play_asset"}
           ] = event_chunk_4.events

    # expected to loose last 30 secs of ad count because of 4 second accumulation of ad deficit
    event_chunk_5 = Enum.at(event_list, 4)
    assert length(event_chunk_5.events) == 4

    assert [
             %{type: "play_asset"},
             %{type: "play_commercial", data: %{"duration" => 34_000}},
             %{type: "play_asset"},
             %{type: "play_asset"}
           ] = event_chunk_5.events
  end

  test "decodes endroll aka pure ad break" do
    insert_assets(5)
    {:ok, cron} = Crontab.CronExpression.Parser.parse("* * * * * *", true)
    schedule_manager = insert(:schedule_manager, %{cron: cron, status: "waiting"})

    csv_rows =
      """
      schedule_block_length,asset_path,commercial_path
      0:00:30,,fallback.ts
      0:01:00,,fallback.ts
      0:06:00,,fallback.ts
      """
      |> csv_string_to_stream
      |> CSV.decode!(headers: true)
      |> Enum.to_list()

    chunk_list = Schedule.to_event_list(csv_rows)

    assert length(chunk_list) == 3
    [%{events: [row1]}, %{events: [row2]}, %{events: [row3]}] = chunk_list
    assert row1.data["duration"] == 34000
    assert row2.data["duration"] == 64000
    assert row3.data["duration"] == 364_000
  end
end
