defmodule MissionControlEx.Web.SchedulePreviewTest do
  use MissionControlEx.Web.ModelCase

  import MissionControlEx.Web.Factory
  import MissionControlEx.Test.TestHelpers
  alias MissionControlEx.Web.{Asset, Event, Schedule, StreamSchedule, ScheduleManager, Repo}

  @linear_csv File.stream!("test/schema/linear_schedule.csv")
  @test_csv File.stream!("test/schema/test_schedule.csv")
  @suspend_csv File.stream!("test/schema/test_suspend_schedule.csv")

  test "preview when stream is mid-chunk" do
    insert_assets_with_commercial_breaks(8)
    schedule_manager = make_schedule_manager(@test_csv)
    start_time = ~N[2017-04-20 16:20:00.0000]

    # Generate 4 events on a manager
    %{schedule_manager: schedule_manager, stream_schedule: stream_schedule} =
      schedule_manager
      |> ScheduleManager.load()
      |> Map.put(:status, "streaming")
      |> ScheduleManager.generate_event_stream(start_time)
      |> Stream.take(4)
      |> Enum.at(-1)

    # Event insertions
    events =
      events =
      Enum.map(schedule_manager.events, fn event ->
        %{event | inserted_at: event.projected_inserted_at}
      end)

    stream_schedule = StreamSchedule.persist_changes(stream_schedule)
    # |> IO.inspect(label: "persisted schedule")

    # Generate fake "current time" for stream
    last_live_at =
      start_time
      |> Timex.add(Timex.Duration.from_milliseconds(Event.duration(schedule_manager.events)))
      |> Timex.add(Timex.Duration.from_seconds(-10))

    # Assert final asset before death was first of chunk 2
    event = Enum.at(schedule_manager.events, -1)
    assert event.data["asset"]["s3_path"] == "2.ts"
    assert event.data["index"] == 0

    # Generate 4 events on a "streaming" manager
    schedule_manager =
      schedule_manager
      |> ScheduleManager.load()
      |> Map.put(:status, "streaming")
      |> Map.put(:events, events)
      |> Map.put(:last_live_at, last_live_at)
      |> ScheduleManager.generate_event_stream()
      |> Stream.take(4)
      |> Enum.at(-1)
      |> Map.get(:schedule_manager)

    # Grab the new events generated above
    preview_events =
      schedule_manager
      |> Map.get(:events)
      |> Enum.take(-4)

    # Asserts first event in preview is correct asset and time
    event = Enum.at(preview_events, 0)
    assert event.data["asset"]["s3_path"] == "2.ts"
    assert event.data["index"] == 1
    assert event.projected_inserted_at == ~N[2017-04-20 16:21:20]

    # Assert second event is generated correctly as well
    event = Enum.at(preview_events, 1)
    assert event.data["asset"]["s3_path"] == "2.ts"
    assert event.data["index"] == 2

    # Assert normal preview after cut off chunk complete
    event = Enum.at(preview_events, 2)
    assert event.data["asset"]["s3_path"] == "3.ts"
    assert event.data["index"] == 0
  end

  test "start from future event chunk" do
    insert_assets(8)
    schedule_manager = make_schedule_manager(@linear_csv)
    start_time = ~N[2017-04-20 16:20:00.0000]

    event_chunk_id =
      schedule_manager
      |> ScheduleManager.generate_event_chunk_stream()
      |> Enum.take(4)
      |> Enum.at(-1)
      |> Map.get(:event_chunk)
      |> Map.get(:id)

    start_here_manager =
      schedule_manager
      |> ScheduleManager.start_from_chunk(event_chunk_id)

    first_chunk =
      start_here_manager
      |> ScheduleManager.update!(%{status: "streaming"})
      |> ScheduleManager.generate_event_chunk_stream(start_time)
      |> Enum.take(1)
      |> Enum.at(0)
      |> Map.get(:event_chunk)

    assert first_chunk.id == event_chunk_id
  end

  test "start from past event chunk" do
    insert_assets(8)
    schedule_manager = make_schedule_manager(@linear_csv)
    start_time = ~N[2017-04-20 16:20:00.0000]

    blocks =
      schedule_manager
      |> ScheduleManager.generate_event_chunk_stream()
      |> Enum.take(4)

    event_chunk_two_id =
      blocks
      |> Enum.at(1)
      |> Map.get(:event_chunk)
      |> Map.get(:id)

    schedule_manager =
      blocks
      |> Enum.at(-1)
      |> Map.get(:schedule_manager)

    stream_schedule =
      schedule_manager
      |> Map.get(:stream_schedules)
      |> Enum.at(0)
      |> StreamSchedule.persist_changes()

    start_here_manager =
      schedule_manager
      |> ScheduleManager.update!(%{status: "streaming"})
      |> ScheduleManager.start_from_chunk(event_chunk_two_id)

    first_chunk =
      start_here_manager
      |> ScheduleManager.generate_event_chunk_stream(start_time)
      |> Enum.take(1)
      |> Enum.at(0)
      |> Map.get(:event_chunk)

    assert first_chunk.id == event_chunk_two_id
  end
end
