defmodule MissionControlEx.Web.ScheduleTest do
  use MissionControlEx.Web.ModelCase

  import MissionControlEx.Web.Factory
  import MissionControlEx.Test.TestHelpers
  alias MissionControlEx.Web.{Asset, Event, Schedule, StreamSchedule, ScheduleManager}

  @linear_csv File.stream!("test/schema/linear_schedule.csv")
  @test_csv File.stream!("test/schema/test_schedule.csv")
  @suspend_csv File.stream!("test/schema/test_suspend_schedule.csv")

  test "gets next event chunk from stream_schedule" do
    insert_assets(8)
    schedule = make_schedule_with_csv(@test_csv)

    {schedule, %{events: [event | _]}} = StreamSchedule.get_next_event_chunk(schedule)
    assert event.data["asset"]["s3_path"] == "1.ts"

    {schedule, _} = StreamSchedule.get_next_event_chunk(schedule)
    {schedule, %{events: events}} = StreamSchedule.get_next_event_chunk(schedule)

    assert Enum.at(events, 0).data["asset"]["s3_path"] == "3.ts"
    assert Enum.at(events, 1).data["asset"]["s3_path"] == "fallback.ts"
  end

  test "generates events with suspend event in middle of schedule" do
    insert_assets(8)
    schedule = make_schedule_with_csv(@suspend_csv)

    {schedule, %{events: [event | _]}} = StreamSchedule.get_next_event_chunk(schedule)
    assert event.data["asset"]["s3_path"] == "1.ts"

    {schedule, _} = StreamSchedule.get_next_event_chunk(schedule)
    {schedule, %{events: [event | _]}} = StreamSchedule.get_next_event_chunk(schedule)

    assert event.type == "suspend"
    assert event.data["duration"] == 1_800_000

    {schedule, %{events: [event | _]}} = StreamSchedule.get_next_event_chunk(schedule)

    assert event.data["asset"]["s3_path"] == "3.ts"
  end

  test "gets correct recovery events when more than one are necessary" do
    insert_assets(8)
    {:ok, cron} = Crontab.CronExpression.Parser.parse("* * * * * *", true)
    schedule_manager = insert(:schedule_manager, %{cron: cron, status: "waiting"})
    stream_schedule = make_schedule_with_csv(@test_csv)

    stream_schedule =
      StreamSchedule.update(stream_schedule, %{schedule_manager_id: schedule_manager.id})

    assert length(stream_schedule.queued_event_chunks) == 7

    start_time = ~N[2017-04-20 16:20:00.0000]

    # Generate 4 events on a manager
    schedule_manager =
      schedule_manager
      |> ScheduleManager.load()
      |> ScheduleManager.generate_event_stream(start_time)
      |> Stream.take(4)
      |> Enum.at(-1)
      |> Map.get(:schedule_manager)

    events =
      Enum.map(schedule_manager.events, fn event ->
        %{event | inserted_at: event.projected_inserted_at}
      end)

    # Generate fake time of stream death
    death_time =
      start_time
      |> Timex.add(Timex.Duration.from_milliseconds(Event.duration(schedule_manager.events)))
      |> Timex.add(Timex.Duration.from_seconds(-60))

    # Generate time to recover at
    recover_time =
      death_time
      |> Timex.add(Timex.Duration.from_seconds(600))

    schedule_manager =
      schedule_manager
      |> Map.put(:status, "recovering")
      |> Map.put(:events, events)
      |> Map.put(:last_live_at, death_time)
      |> ScheduleManager.generate_event_stream(recover_time)
      |> Stream.take(3)
      |> Enum.at(-1)
      |> Map.get(:schedule_manager)

    # Get recovery events
    recovery_events =
      schedule_manager
      |> Map.get(:events)
      |> Enum.take(-3)

    # Black buffer on front
    event = Enum.at(recovery_events, 0)
    assert event.data["asset"]["s3_path"] == nil
    assert event.type == "stream_black_buffer"

    # Assert play_asset event was recovered and started at offset time
    event = Enum.at(recovery_events, 1)
    assert event.data["asset"]["s3_path"] == "fallback.ts"
    assert event.data["start_time"] == 4000

    # Assert cut off commercial was recovered and started at original start time
    event = Enum.at(recovery_events, 2)
    assert event.data["asset"]["s3_path"] == "4.ts"
    assert event.data["start_time"] == 0
  end

  test "recovers correctly when death is mid chunk" do
    insert_assets_with_commercial_breaks(8)
    {:ok, cron} = Crontab.CronExpression.Parser.parse("* * * * * *", true)
    schedule_manager = insert(:schedule_manager, %{cron: cron, status: "waiting"})
    stream_schedule = make_schedule_with_csv(@test_csv)

    stream_schedule =
      StreamSchedule.update(stream_schedule, %{schedule_manager_id: schedule_manager.id})

    assert length(stream_schedule.queued_event_chunks) == 7

    start_time = ~N[2017-04-20 16:20:00.0000]

    # Generate 4 events on a manager
    %{schedule_manager: schedule_manager, stream_schedule: stream_schedule} =
      schedule_manager
      |> ScheduleManager.load()
      |> ScheduleManager.generate_event_stream(start_time)
      |> Stream.take(8)
      |> Enum.at(-1)

    # Event insertions
    events =
      events =
      Enum.map(schedule_manager.events, fn event ->
        %{event | inserted_at: event.projected_inserted_at}
      end)

    stream_schedule = StreamSchedule.persist_changes(stream_schedule)

    # Generate fake time of stream death
    # 1 second left in commerical
    death_time =
      start_time
      |> Timex.add(Timex.Duration.from_milliseconds(Event.duration(schedule_manager.events)))
      |> Timex.add(Timex.Duration.from_seconds(-3))

    # Generate time to recover at
    recover_time =
      death_time
      |> Timex.add(Timex.Duration.from_seconds(600))

    %{schedule_manager: schedule_manager, stream_schedule: stream_schedule} =
      schedule_manager
      |> ScheduleManager.load()
      |> Map.put(:status, "recovering")
      |> Map.put(:events, events)
      |> Map.put(:last_live_at, death_time)
      |> ScheduleManager.generate_event_stream(recover_time)
      |> Stream.take(3)
      |> Enum.at(-1)

    # Get recovery events
    recovery_events =
      schedule_manager
      |> Map.get(:events)
      |> Enum.take(-2)

    # Assert cut off commercial was recovered and started at delayed start time
    event = Enum.at(recovery_events, 0)
    assert event.data["asset"]["s3_path"] == "fallback.ts"
    assert event.data["start_time"] == 1_000

    # Assert next event is still within chunk
    event = Enum.at(recovery_events, 1)
    assert event.data["asset"]["s3_path"] == "3.ts"
    assert event.data["index"] == 1
  end

  test "recovers correctly when death is mid suspend" do
    insert_assets_with_commercial_breaks(8)
    {:ok, cron} = Crontab.CronExpression.Parser.parse("* * * * * *", true)
    schedule_manager = insert(:schedule_manager, %{cron: cron, status: "waiting"})
    stream_schedule = make_schedule_with_csv(@suspend_csv)

    stream_schedule =
      StreamSchedule.update(stream_schedule, %{schedule_manager_id: schedule_manager.id})

    assert length(stream_schedule.queued_event_chunks) == 8

    start_time = ~N[2017-04-20 16:20:00.0000]

    # Generate 4 events on a manager
    schedule_manager =
      schedule_manager
      |> ScheduleManager.load()
      |> ScheduleManager.generate_event_stream(start_time)
      |> Stream.take(7)
      |> Enum.at(-1)
      |> Map.get(:schedule_manager)

    events =
      Enum.map(schedule_manager.events, fn event ->
        %{event | inserted_at: event.projected_inserted_at}
      end)

    # Generate fake time of stream death
    death_time =
      start_time
      |> Timex.add(Timex.Duration.from_milliseconds(Event.duration(schedule_manager.events)))
      |> Timex.add(Timex.Duration.from_seconds(-20 * 60))

    # |> IO.inspect(label: "death time")

    # Generate time to recover at
    recover_time =
      death_time
      |> Timex.add(Timex.Duration.from_seconds(3600 * 1000))

    schedule_manager =
      schedule_manager
      |> Map.put(:status, "recovering")
      |> Map.put(:events, events)
      |> Map.put(:last_live_at, death_time)
      |> ScheduleManager.generate_event_stream(recover_time)
      |> Stream.take(3)
      |> Enum.at(-1)
      |> Map.get(:schedule_manager)

    # Get recovery events
    recovery_events =
      schedule_manager
      |> Map.get(:events)
      |> Enum.take(-5)

    [
      previous_played_event,
      interrupted_suspend,
      recover_suspend,
      black_frames,
      next_recover_event
    ] = recovery_events

    assert previous_played_event.type == "play_asset"
    assert previous_played_event.type == "play_asset"
    assert interrupted_suspend.type == "suspend"

    assert recover_suspend.data["recover"]
    assert recover_suspend.type == "suspend"

    assert recover_suspend.data["duration"] + recover_suspend.data["start_time"] ==
             interrupted_suspend.data["duration"]

    assert black_frames.type == "stream_black_buffer"
    assert black_frames.data["duration"] == 31_000

    assert next_recover_event.type == "play_asset"
    assert next_recover_event.data["asset"]["s3_path"] == "3.ts"
    assert next_recover_event.data["index"] == 0
  end

  test "gets correct assets when recovering a start failure" do
    insert_assets(8)
    {:ok, cron} = Crontab.CronExpression.Parser.parse("* * * * * *", true)
    schedule_manager = insert(:schedule_manager, %{cron: cron, status: "waiting"})
    stream_schedule = make_schedule_with_csv(@test_csv)

    stream_schedule =
      StreamSchedule.update(stream_schedule, %{schedule_manager_id: schedule_manager.id})

    assert length(stream_schedule.queued_event_chunks) == 7

    start_time = ~N[2017-04-20 16:20:00.0000]

    # Generate 4 events on a manager
    schedule_manager =
      schedule_manager
      |> ScheduleManager.load()
      |> Map.put(:status, "recovering")
      |> ScheduleManager.generate_event_stream(start_time)
      |> Stream.take(4)
      |> Enum.at(-1)
      |> Map.get(:schedule_manager)

    # Asserts status is as expected
    assert schedule_manager.status == "streaming"
    assert schedule_manager.last_live_at == nil

    # Gets first asset
    event = Enum.at(schedule_manager.events, 0)
    assert event.data["asset"]["s3_path"] == "1.ts"

    # Gets second asset
    event = Enum.at(schedule_manager.events, 1)
    assert event.data["asset"]["s3_path"] == "2.ts"
  end

  test "schedule reset on end and return at correct time" do
    insert_assets(8)
    {:ok, cron} = Crontab.CronExpression.Parser.parse("0 0 18 * * *", true)
    schedule_manager = insert(:schedule_manager, %{cron: cron, status: "waiting"})
    stream_schedule = make_schedule_with_csv(@test_csv)

    stream_schedule =
      StreamSchedule.update(stream_schedule, %{
        schedule_manager_id: schedule_manager.id,
        repeat: true
      })

    start_time = ~N[2017-04-20 16:20:00]

    schedule_manager =
      schedule_manager
      |> ScheduleManager.load()
      |> ScheduleManager.generate_event_stream(start_time)
      |> Stream.take(13)
      |> Stream.each(fn %{event: event} -> Event.process_persist_update(event) end)
      |> Enum.at(-1)
      |> Map.get(:schedule_manager)

    # Assert that current_index is nil after stream end
    current_index =
      schedule_manager
      |> Map.get(:stream_schedules)
      |> Enum.at(0)
      |> Map.get(:chunk_index)

    assert current_index == nil

    schedule_manager =
      schedule_manager
      |> Map.put(:last_started_at, start_time)
      |> ScheduleManager.generate_event_stream()
      |> Stream.take(2)
      |> Stream.each(fn %{event: event} -> Event.process_persist_update(event) end)
      |> Enum.at(-1)
      |> Map.get(:schedule_manager)

    # First asset after end_stream is correct asset and time
    event = Enum.at(schedule_manager.events, -2)
    assert event.data["asset"]["s3_path"] == "1.ts"
    assert event.projected_inserted_at == ~N[2017-04-20 18:00:00]

    # Second asset after end_stream is correct asset and time
    event = Enum.at(schedule_manager.events, -1)
    assert event.data["asset"]["s3_path"] == "2.ts"
    assert event.projected_inserted_at == ~N[2017-04-20 18:01:00]
  end

  test "schedule ends, resets, and stops (Cosmos)" do
    insert_assets(8)
    {:ok, cron} = Crontab.CronExpression.Parser.parse("* * * * * *", true)
    schedule_manager = insert(:schedule_manager, %{cron: cron, status: "waiting"})
    stream_schedule = make_schedule_with_csv(@test_csv)

    stream_schedule =
      StreamSchedule.update(stream_schedule, %{
        schedule_manager_id: schedule_manager.id,
        repeat: true
      })

    assert length(stream_schedule.queued_event_chunks) == 7

    start_time = ~N[2017-04-20 16:20:00.0000]

    # take events until event_list is empty
    %StreamEventChunkBlock{
      schedule_manager: schedule_manager,
      stream_schedule: done_schedule,
      event_chunk: event_chunk
    } =
      schedule_manager
      |> ScheduleManager.load()
      |> ScheduleManager.generate_event_chunk_stream(start_time)
      |> Stream.take(8)
      |> Enum.at(-1)

    assert [%Event{type: "end_stream"}] = event_chunk.events
    assert done_schedule.played_event_chunks == []
    assert length(done_schedule.queued_event_chunks) == 7

    # pretend all events played
    events =
      Enum.map(schedule_manager.events, fn event ->
        Event.process_persist_update(event)
      end)

    first_end_time = List.last(events).inserted_at
    next_start_time = ScheduleManager.get_start_time(schedule_manager)

    %StreamEventChunkBlock{
      schedule_manager: schedule_manager,
      stream_schedule: done_schedule,
      event_chunk: event_chunk
    } =
      schedule_manager
      |> ScheduleManager.load()
      |> Map.put(:status, "waiting")
      |> Map.put(:events, events)
      |> Map.put(:last_live_at, first_end_time)
      |> Map.put(:last_started_at, start_time)
      |> ScheduleManager.generate_event_chunk_stream(next_start_time)
      |> Stream.take(8)
      |> Enum.at(-1)

    # second time around
    assert [%Event{type: "end_stream"}] = event_chunk.events
    assert done_schedule.played_event_chunks == []
    assert length(done_schedule.queued_event_chunks) == 7
  end

  test "schedule repeats forever" do
    insert_assets(8)
    {:ok, cron} = Crontab.CronExpression.Parser.parse("* * * * * *", true)
    schedule_manager = insert(:schedule_manager, %{cron: cron, status: "waiting"})
    stream_schedule = make_schedule_with_csv(@test_csv)

    stream_schedule =
      StreamSchedule.update(stream_schedule, %{
        schedule_manager_id: schedule_manager.id,
        repeat: true,
        stop_on_empty: false
      })

    assert length(stream_schedule.queued_event_chunks) == 7

    start_time = ~N[2017-04-20 16:20:00.0000]

    # take events until event_list is empty
    %StreamEventChunkBlock{
      schedule_manager: schedule_manager,
      stream_schedule: reset_schedule,
      event_chunk: event_chunk
    } =
      schedule_manager
      |> ScheduleManager.load()
      |> ScheduleManager.generate_event_chunk_stream(start_time)
      |> Stream.take(7)
      |> Enum.at(-1)

    assert [%Event{data: %{"asset" => %{"s3_path" => "7.ts"}}} | _] = event_chunk.events
    assert length(reset_schedule.played_event_chunks) == 7
    assert reset_schedule.queued_event_chunks == []

    # pretend all events played
    events =
      Enum.map(schedule_manager.events, fn event ->
        Event.process_persist_update(event)
      end)

    first_end_time = List.last(events).inserted_at
    next_start_time = ScheduleManager.get_start_time(schedule_manager)

    %StreamEventChunkBlock{
      schedule_manager: schedule_manager,
      stream_schedule: reset_schedule,
      event_chunk: event_chunk
    } =
      schedule_manager
      |> ScheduleManager.load()
      |> ScheduleManager.generate_event_chunk_stream(next_start_time)
      |> Stream.take(7)
      |> Enum.at(-1)

    # second time around
    assert [%Event{data: %{"asset" => %{"s3_path" => "7.ts"}}} | _] = event_chunk.events
    assert length(reset_schedule.played_event_chunks) == 7
    assert reset_schedule.queued_event_chunks == []
  end

  test "schedule plays for correct length" do
    insert_assets_with_commercial_breaks(5)
    {:ok, cron} = Crontab.CronExpression.Parser.parse("0 0 18 * * *", true)
    schedule_manager = insert(:schedule_manager, %{cron: cron, status: "waiting"})

    csv_stream =
      """
      schedule_block_length,asset_path,commercial_path
      0:02:00,1.ts,fallback.ts
      0:19:00,2.ts,fallback.ts
      0:00:30,3.ts,fallback.ts
      0:04:00,4.ts,fallback.ts
      ,5.ts,fallback.ts
      """
      |> csv_string_to_stream

    stream_schedule = make_schedule_with_csv(csv_stream)

    stream_schedule =
      StreamSchedule.update(stream_schedule, %{
        schedule_manager_id: schedule_manager.id,
        repeat: false
      })

    assert length(stream_schedule.queued_event_chunks) == 5

    start_time = ~N[2017-04-20 16:20:00.0000]

    # take events until event_list is empty
    %StreamEventBlock{
      schedule_manager: schedule_manager,
      stream_schedule: done_schedule,
      event: event
    } =
      schedule_manager
      |> ScheduleManager.load()
      |> ScheduleManager.generate_event_stream(start_time)
      |> Stream.take(24)
      |> Enum.at(-1)

    assert event.type == "end_stream"
    # 2 min + 19 min + 1 min + 4 min (3.ts gets fully played)
    assert Timex.diff(event.projected_inserted_at, start_time, :minutes) == 26
  end

  test "refill and suspend a schedule with a duration" do
    insert_assets(8)

    schedule_manager =
      make_schedule_manager(@linear_csv, %{duration: 60_000, stop_on_empty: true})

    start_time = ~N[2017-04-20 16:20:00]
    schedule_manager = ScheduleManager.update!(schedule_manager, %{cron: "0 */2 * * * *"})

    schedule_manager =
      schedule_manager
      |> ScheduleManager.load()
      |> Map.put(:last_started_at, start_time)
      |> ScheduleManager.generate_event_stream(start_time)
      |> Stream.take(8)
      |> Enum.at(-1)
      |> Map.get(:schedule_manager)

    events = Map.get(schedule_manager, :events)

    # Assert first end occurs correctly at right time given duration
    event = Enum.at(events, 1)
    assert event.type == "end_stream"
    assert event.projected_inserted_at == ~N[2017-04-20 16:21:00]

    # Assert schedule resumes at correct time given cron
    event = Enum.at(events, 2)
    assert event.type == "play_asset"
    assert event.projected_inserted_at == ~N[2017-04-20 18:00:00]

    # Assert time still in sync after multiple ends and resumes
    event = Enum.at(events, 6)
    assert event.type == "play_asset"
    assert event.projected_inserted_at == ~N[2017-04-20 22:00:00]
    event = Enum.at(events, 7)
    assert event.type == "end_stream"
    assert event.projected_inserted_at == ~N[2017-04-20 22:01:00]
  end

  test "repeat a schedule with a duration" do
    insert_assets(8)

    schedule_manager =
      make_schedule_manager(@linear_csv, %{duration: 60_000, stop_on_empty: true, repeat: true})

    start_time = ~N[2017-04-20 16:20:00]
    schedule_manager = ScheduleManager.update!(schedule_manager, %{cron: "0 */2 * * * *"})

    schedule_manager =
      schedule_manager
      |> ScheduleManager.load()
      |> Map.put(:last_started_at, start_time)
      |> ScheduleManager.generate_event_stream(start_time)
      |> Stream.take(25)
      |> Enum.at(-1)
      |> Map.get(:schedule_manager)

    events = Map.get(schedule_manager, :events)
    # Assert first end occurs correctly at right time given duration
    event = Enum.at(events, 14)
    assert event.type == "play_asset"
    assert event.data["asset"]["s3_path"] == "1.ts"
    assert event.projected_inserted_at == ~N[2017-04-21 06:00:00]
  end

  test "repeat correctly with duration when filling from too few event_chunks" do
    insert_assets(8)

    schedule_manager =
      make_schedule_manager(@linear_csv, %{duration: 120_000, stop_on_empty: true, repeat: true})

    start_time = ~N[2017-04-20 16:20:00]
    schedule_manager = ScheduleManager.update!(schedule_manager, %{cron: "0 */2 * * * *"})

    schedule_manager =
      schedule_manager
      |> ScheduleManager.load()
      |> Map.put(:last_started_at, start_time)
      |> ScheduleManager.generate_event_stream(start_time)
      |> Stream.take(25)
      |> Enum.at(-1)
      |> Map.get(:schedule_manager)

    Event.print_events(schedule_manager)

    events = Map.get(schedule_manager, :events)

    # Assert first play after empty is correct
    event = Enum.at(events, 10)
    assert event.type == "play_asset"
    assert event.data["asset"]["s3_path"] == "1.ts"

    # Assert next set after empty is correct
    event = Enum.at(events, 12)
    assert event.type == "play_asset"
    assert event.data["asset"]["s3_path"] == "2.ts"

    event = Enum.at(events, 13)
    assert event.type == "play_asset"
    assert event.data["asset"]["s3_path"] == "3.ts"
  end
end
