defmodule MissionControlEx.Web.ScheduleValidationTest do
  use MissionControlEx.Web.ModelCase

  import MissionControlEx.Web.Factory

  alias MissionControlEx.Web.{
    Asset,
    Event,
    Schedule,
    StreamSchedule,
    ScheduleManager,
    Schedule.ScheduleInvalidError
  }

  @empty []

  def insert_assets(num) do
    for x <- 1..num, do: insert(:asset, %{s3_path: "#{x}.ts", duration: 6 * 10_000})
    insert(:asset, %{s3_path: "fallback.ts", duration: 30_000})
  end

  def insert_assets_with_commercial_breaks(num) do
    for x <- 1..num do
      insert(:asset, %{
        s3_path: "#{x}.ts",
        commercial_breaks_ms: [20_000, 50_000],
        duration: 6 * 10_000
      })
    end

    insert(:asset, %{s3_path: "fallback.ts", duration: 120_000})
  end

  def csv_string_to_rows(csv_string) do
    {:ok, pid} = StringIO.open(csv_string)

    pid
    |> IO.stream(:line)
    |> CSV.decode!(headers: true)
    |> Enum.to_list()
  end

  test "raises if schedule headers are invalid" do
    invalid_csv =
      """
      wrong_headers,;DROP TABLES;,;;;DROP SCHEMA public CASCADE;
      0:02:00,1.ts,fallback.ts
      """
      |> csv_string_to_rows

    assert_raise(
      MissionControlEx.Web.Schedule.ScheduleInvalidError,
      ~r/^missing required headers/,
      fn -> Schedule.to_event_list(invalid_csv) end
    )
  end

  test "invalidates schedule with incomplete or nonexistant assets" do
    insert(:asset, %{s3_path: "complete.ts", duration: 30_000})
    insert(:asset, %{s3_path: "incomplete.ts", status: "transcoding"})
    insert(:asset, %{s3_path: "not_done.ts", status: "ts_convert"})

    invalid_csv =
      """
      schedule_block_length,asset_path,commercial_path
      0:02:00,complete.ts,not_done.ts
      0:02:00,incomplete.ts,
      0:02:00,incomplete.ts,not_done.ts
      0:02:00,does_not_exist.ts,
      """
      |> csv_string_to_rows

    error =
      assert_raise(MissionControlEx.Web.Schedule.ScheduleInvalidError, ~r/^Invalid Rows/, fn ->
        Schedule.to_event_list(invalid_csv)
      end)

    assert length(error.invalid_rows) == 4
    assert [row1, row2, row3, row4] = error.invalid_rows
    assert String.contains?(row1, ["Row1", "not_done.ts"])
    assert String.contains?(row2, ["Row2", "incomplete.ts"])
    assert String.contains?(row3, ["Row3", "incomplete.ts", "not_done.ts"])
    assert String.contains?(row4, ["Row4", "does_not_exist.ts"])
  end

  test "invalidates schedule with no time for suspend" do
    insert(:asset, %{s3_path: "1.ts", duration: 30_000, status: "complete"})

    invalid_csv =
      """
      schedule_block_length,asset_path,commercial_path
      0:02:00,1.ts,
      1:00:00,suspend,
      0:02:00,1.ts,
      ,suspend,
      """
      |> csv_string_to_rows

    error =
      assert_raise(MissionControlEx.Web.Schedule.ScheduleInvalidError, ~r/^Invalid Rows/, fn ->
        Schedule.to_event_list(invalid_csv)
      end)

    assert length(error.invalid_rows) == 1
    assert [row4] = error.invalid_rows
    assert String.contains?(row4, ["Row4", "Invalid Block Length", "hh:mm:ss"])
  end

  test "commercial breaks exceed maximum" do
    insert(:asset, %{s3_path: "1.ts", duration: 60_000, commercial_breaks_ms: [30_000]})

    insert(:asset, %{
      s3_path: "ad.ts",
      duration: 10 * 60_000,
      commercial_breaks_ms: [20_000, 50_000]
    })

    invalid_csv =
      """
      schedule_block_length,asset_path,commercial_path
      0:15:00,1.ts,ad.ts
      """
      |> csv_string_to_rows

    assert_raise(ScheduleInvalidError, ~r/not enough ad breaks/, fn ->
      Schedule.to_event_list(invalid_csv)
    end)
  end
end
