ExUnit.start()

# Create the database, run migrations, and start the test transaction.
Mix.Task.run("ecto.create", ["--quiet"])
Mix.Task.run("ecto.migrate", ["--quiet"])
Ecto.Adapters.SQL.Sandbox.mode(MissionControlEx.Web.Repo, :manual)

defmodule MissionControlEx.Test.TestHelpers do
  use MissionControlEx.Web.ModelCase
  import MissionControlEx.Web.Factory
  alias MissionControlEx.Web.{Router, Repo, Event, StreamSchedule, ScheduleManager, Asset, Event}

  def insert_assets(num) do
    for x <- 1..num do
      insert(:asset, %{s3_path: "#{x}.ts", duration: 6 * 10_000})
    end

    insert(:asset, %{s3_path: "fallback.ts", duration: 30_000})
  end

  def insert_assets_with_commercial_breaks(num) do
    for x <- 1..num do
      insert(:asset, %{
        s3_path: "#{x}.ts",
        commercial_breaks_ms: [20_000, 50_000],
        duration: 6 * 10_000
      })
    end

    insert(:asset, %{s3_path: "fallback.ts", duration: 120_000})
  end

  def make_schedule_with_csv(csv, options \\ %{}) do
    options = Map.merge(%{state: "waiting", name: "test", default_schedule: true}, options)

    stream_schedule =
      insert(:stream_schedule, options)
      |> StreamSchedule.load_event_list_from_csv(csv)
      |> StreamSchedule.persist_changes()
  end

  def csv_string_to_stream(csv_string) do
    {:ok, pid} = StringIO.open(csv_string)
    pid |> IO.stream(:line)
  end

  def make_schedule_manager(csv, options \\ %{}) do
    {:ok, cron} = Crontab.CronExpression.Parser.parse("* * * * * *", true)
    schedule_manager = insert(:schedule_manager, %{cron: cron, status: "waiting"})
    stream_schedule = make_schedule_with_csv(csv, options)

    stream_schedule =
      StreamSchedule.update(stream_schedule, %{schedule_manager_id: schedule_manager.id})

    ScheduleManager.load(schedule_manager)
  end

  def print_events_in_stream_schedule(
        %{played_event_chunks: played, queued_event_chunks: to_play} = stream_schedule
      ) do
    if played != [] do
      IO.puts("Played Events")

      Enum.flat_map(played, & &1.events)
      |> Event.do_print_events()
    else
      IO.puts("Played Events: []")
    end

    if to_play != [] do
      IO.puts("Event Chunks")

      Enum.flat_map(to_play, & &1.events)
      |> Event.do_print_events()
    else
      IO.puts("Events: []")
    end
  end
end
