class StreamSession < ApplicationRecord
  include AASM

  just_define_datetime_picker :start_time, :add_to_attr_accessible => false

  belongs_to :channel
  has_many :stream_session_videos, -> { order("\"order\" asc") }
  has_many :videos, through: :stream_session_videos

  has_one :current_stream_session_video, -> { where(state: "streaming") }, class_name: "StreamSessionVideo"
  has_one :next_stream_session_video, -> { where(state: "unstreamed").order("\"order\" asc") }, class_name: "StreamSessionVideo"

  aasm column: 'state', whiny_transitions: false do
    state :scheduled, initial: true
    state :started
    state :canceled
    state :finished

    event :finish do
      transitions from: :started, to: :finished, guard: :stream_session_videos_finished?
    end

    event :cancel do
      transitions from: [:scheduled, :started, :finished], to: :canceled
    end
  end

  def deep_duplicate
    ss = StreamSession.new(self.as_json(except: :id, root: false))
    ss.save!
    ss.stream_session_videos.create(self.stream_session_videos.as_json(except:[:id, :stream_session_id], root: false))
    ss
  end

  def start_time
    super || Time.now
  end

  def stream_session_videos_finished?
    stream_session_videos.all? { |ssv| ssv.finished? }
  end

  def ensure_indefinite_stream_session
    if self.indefinite && !stream_session_videos.any?(&:unstreamed)
      exclusions = ordered_stream_session_videos.map(&:video)[-20..-1] || []
      video = Video.get_random(channel, exclusions: exclusions)[0]
      if video.present?
        prepend_video(video)
      else
        Rollbar.critical("Could not add new video to indefinite stream!", stream_session_id: id)
      end
    end
  end

  def ordered_stream_session_videos
    stream_session_videos.includes(:video).sort_by { |ssv| ssv.order }
  end

  def ordered_videos
    ordered_stream_session_videos.map(&:video)
  end

  def possible_videos
    channel.videos
  end

  def append_video(video)
    transaction do
      order = (stream_session_videos.map(&:order).max || 0) + 1
      self.stream_session_videos.create!(video_id: video.id, seconds_streamed: 0, order: order)
    end
  end

  def prepend_video(video)
    transaction do
      unstreamed_session_videos = self.unstreamed_session_videos
      last_stream_session_video = stream_session_videos.last
      order = current_stream_session_video ? current_stream_session_video.order + 1 : (unstreamed_session_videos.map(&:order).min || (last_stream_session_video.try(:order) || 0) + 1)
      unstreamed_session_videos.each { |ssv| ssv.update_attributes(order: ssv.order + 1 ) }
      self.stream_session_videos.create!(video_id: video.id, seconds_streamed: 0, order: order)
    end
  end

  def unstreamed_session_videos
    self.stream_session_videos.select {|ssv| ssv.state == "unstreamed"}
  end

  def interleave_video(video)
    transaction do
      unstreamed_session_videos = self.unstreamed_session_videos
      starting_from = (unstreamed_session_videos.map(&:order).min || 1)
      interleaved_session_videos = (unstreamed_session_videos.count - 1).times.map do
        StreamSessionVideo.new(stream_session_id: id, video_id: video.id)
      end
      unstreamed_session_videos.zip(interleaved_session_videos).flatten.compact.each_with_index do |ssv, i|
        ssv.order = starting_from + i
        ssv.save!
      end
    end
  end

  def self.pop
    transaction do
      result = connection.execute(
      <<-SQL
          lock table stream_sessions in exclusive mode;
        update
            stream_sessions
        set
            state = 'started', updated_at = '#{0.minutes.ago}'
        where
            id in (
                select
                    id
                from
                    stream_sessions
                where
                    start_time <= '#{Time.now}' and
                    (
                      state = 'scheduled' or
                      (state = 'started' and updated_at < '#{1.minutes.ago}')
                    )
                order by
                    id asc
                limit 1
            )
        returning id;
      SQL
      )
      id = result.to_a.first.try(:[], "id")
      if id.present?
        StreamSession.find(id)
      end
    end
  end
end
