import React, { Component } from 'react';
import _ from 'lodash';
import { Link } from 'react-router';
import { desktopCapturer } from 'electron';
import Draggable from 'react-draggable';
import { exec } from 'child-process-promise';
import yaml from 'js-yaml';
import styles from './Step2.css';
import { fileUpdate } from '../utils/obs';

const OBS_HEIGHT = 1440;
const OBS_WIDTH = 2560;

const WEB_CAM_RESOLUTIONS = [
    { width: 1920, height: 1080 },
    { width: 1280, height: 720 },
    { width: 960, height: 720 },
    { width: 640, height: 360 },
    { width: 640, height: 480 },
    { width: 320, height: 240 },
    { width: 320, height: 180 }
];

export default class Step2 extends Component {
  constructor(props) {
    super(props);
    this.state = {
      videoDevices: [],
      hoverOverMainPicture: false,
      hoverOverSmallPicture: false,
      smallPicturePreview: null,
      mainPicturePreview: null,
      smallPicture: null,
      mainPicture: null,
      currentVideo: null
    };
    this.bootStrapVideoSources();
  }

  componentDidUpdate() {
    Array.prototype.forEach.call(document.getElementsByTagName('video'), (v) => {
      if (v.paused && v.src !== '') v.play();
    });
  }


  bootStrapVideoSources() {
    navigator.mediaDevices.enumerateDevices().then((devices) =>
      Promise.all(devices.filter((d) => d.kind === 'videoinput').map((d) =>
      WEB_CAM_RESOLUTIONS.reduce((accum, res) =>
        accum.catch(() => navigator.mediaDevices.getUserMedia(
          {
            video: {
              mandatory: {
                sourceId: d.deviceId,
                minWidth: res.width,
                maxWidth: res.width,
                minHeight: res.height,
                maxHeight: res.height
              }
            }
          }
        )
      ), Promise.reject())
      ))
    ).then((streams) => {
      const videoDevices = streams.map((stream) => {
        const fakeVideoElement = document.createElement('video');
        fakeVideoElement.src = URL.createObjectURL(stream);
        fakeVideoElement.addEventListener('canplay', function(){
          const width = fakeVideoElement.videoWidth;
          const height = fakeVideoElement.videoHeight;
          this.setState({ videoDevices: this.state.videoDevices.concat([{ type: 'webcam', url: URL.createObjectURL(stream), stream, width, height }]) });
        }.bind(this))
      });
    }).then(() =>
      desktopCapturer.getSources({ types: ['screen'] }, (_, sources) => {
        sources.reduce((promise, source) => {
          return promise.then(() =>
          navigator.mediaDevices.getUserMedia({
            audio: false,
            video: {
              mandatory: {
                chromeMediaSource: 'desktop',
                chromeMediaSourceId: source.id,
                minWidth: 1280,
                maxWidth: 1280,
                minHeight: 720,
                maxHeight: 720
              }
            }
          }).then((stream) =>
            this.setState({ videoDevices: this.state.videoDevices.concat([{ type: 'screen', screenNumber: parseInt(source.name.slice(7), 10), url: URL.createObjectURL(stream), width: 2560, height: 1440, stream }])}))
          )
        }, Promise.resolve());
      })
    ).then(() => exec('system_profiler SPCameraDataType')
    ).then((p) => {
      let payload = p.stdout.replace(/(\n +)(.+):\n/g, "$1\"$2\":\n").replace(/(0x.+)/g, "\"$1\"")
      const camLookup = yaml.safeLoad(payload, 'utf8').Camera
      const keys = _.sortBy(Object.keys(camLookup), (label) => -label.length);
      const videoDevices = this.state.videoDevices.map((v) => {
        const name = v.stream.getVideoTracks()[0].label;
        const key = keys.find((k) => name.startsWith(k));
        if (key) {
          const video = Object.assign({ obsId: camLookup[key]['Unique ID'], obsName: key }, v);
          return video;
        }
        return v;
      });
      return this.setState({ videoDevices });
    }).catch((err) => console.log(err));
  }

  hoverOverMainPicture() {
    if (this.state.currentVideo) {
      this.setState({
        mainPicturePreview: this.state.currentVideo,
        hoverOverMainPicture: true,
      });
    }
  }

  leaveMainPicture() {
    if (this.state.currentVideo) {
      this.setState({
        mainPicturePreview: null,
        hoverOverMainPicture: false,
      });
    }
  }


  hoverOverSmallPicture() {
    if (this.state.currentVideo) {
      this.setState({
        smallPicturePreview: this.state.currentVideo,
        hoverOverSmallPicture: true,
      });
    }
  }

  leaveSmallPicture() {
    if (this.state.currentVideo) {
      this.setState({
        smallPicturePreview: null,
        hoverOverSmallPicture: false,
      });
    }
  }

  persistPreview() {
    this.setState({
      smallPicture: this.state.smallPicturePreview || this.state.smallPicture,
      mainPicture: this.state.mainPicturePreview || this.state.mainPicture,
      mainPicturePreview: null,
      smallPicturePreview: null,
      currentVideo: null,
    });
    try { this.saveLayout(); } catch (e) { console.log(e); }
  }

  selectVideoInput(currentVideo) {
    this.setState({ currentVideo });
  }

  deleteMainPicture() {
    this.setState({ mainPicture: null });
  }

  deleteSmallPicture() {
    this.setState({ smallPicture: null });
  }

  obsSettingsFor(input) {
    switch (input.type) {
      case 'webcam': {
        return {
          "device": input.obsId,
          "device_name": input.obsName
        };
      }
      case 'screen': {
        return {
          "display": input.screenNumber - 1
        };
      }
      default: {
        throw new Error(`Type ${input.type} invalid`);
      }
    }
  }

  // Enforces 16:9
  enforceObsDimensions({ height, width }) {
    const sourceRatio = width / height;
    let scale;
    let x;
    let y;
    if (sourceRatio > OBS_WIDTH / OBS_HEIGHT) {
      scale = OBS_WIDTH / width;
    } else {
      scale = OBS_HEIGHT / height;
    }
    const scaledWidth = scale * width;
    const scaledHeight = scale * height;

    if (sourceRatio > OBS_WIDTH / OBS_HEIGHT) {
      x = 0;
      y = (OBS_HEIGHT - scaledHeight) / 2;
    } else {
      x = (OBS_WIDTH - scaledWidth) / 2;
      y = 0;
    }
    return { x, y, scale };
  }

  obsSources(mainPicture, smallPicture) {
    // Force 16:9
    const mainPictureDimensions = this.enforceObsDimensions(mainPicture);
    const smallPictureDimensions = this.enforceObsDimensions(smallPicture);
    smallPictureDimensions.scale *= 0.25;
    smallPictureDimensions.x *= 0.25;
    smallPictureDimensions.y *= 0.25;
    smallPictureDimensions.y += (OBS_HEIGHT * (1 - 0.25));
    return [
        {
            "deinterlace_field_order": 0,
            "deinterlace_mode": 0,
            "enabled": true,
            "flags": 0,
            "hotkeys": {},
            "id": mainPicture.type === 'webcam' ? "av_capture_input" : "display_capture",
            "mixers": 0,
            "muted": false,
            "name": "Main Picture",
            "push-to-mute": false,
            "push-to-mute-delay": 0,
            "push-to-talk": false,
            "push-to-talk-delay": 0,
            "settings": this.obsSettingsFor(mainPicture),
            "sync": 0,
            "volume": 1.0
        },
        {
            "deinterlace_field_order": 0,
            "deinterlace_mode": 0,
            "enabled": true,
            "flags": 0,
            "hotkeys": {
                "OBSBasic.SelectScene": [],
                "libobs.hide_scene_item.Main Picture": [],
                "libobs.hide_scene_item.Small Picture": [],
                "libobs.show_scene_item.Main Picture": [],
                "libobs.show_scene_item.Small Picture": []
            },
            "id": "scene",
            "mixers": 0,
            "muted": false,
            "name": "Scene",
            "push-to-mute": false,
            "push-to-mute-delay": 0,
            "push-to-talk": false,
            "push-to-talk-delay": 0,
            "settings": {
                "items": [
                    {
                        "align": 5,
                        "bounds": {
                            "x": 0.0,
                            "y": 0.0
                        },
                        "bounds_align": 0,
                        "bounds_type": 0,
                        "crop_bottom": 0,
                        "crop_left": 0,
                        "crop_right": 0,
                        "crop_top": 0,
                        "name": "Main Picture",
                        "pos": {
                            "x": mainPictureDimensions.x,
                            "y": mainPictureDimensions.y
                        },
                        "rot": 0.0,
                        "scale": {
                            "x": mainPictureDimensions.scale,
                            "y": mainPictureDimensions.scale
                        },
                        "scale_filter": "disable",
                        "visible": true
                    },
                    {
                        "align": 5,
                        "bounds": {
                            "x": 0.0,
                            "y": 0.0
                        },
                        "bounds_align": 0,
                        "bounds_type": 0,
                        "crop_bottom": 0,
                        "crop_left": 0,
                        "crop_right": 0,
                        "crop_top": 0,
                        "name": "Small Picture",
                        "pos": {
                            "x": smallPictureDimensions.x,
                            "y": smallPictureDimensions.y
                        },
                        "rot": 0.0,
                        "scale": {
                            "x": smallPictureDimensions.scale,
                            "y": smallPictureDimensions.scale
                        },
                        "scale_filter": "disable",
                        "visible": true
                    }
                ]
            },
            "sync": 0,
            "volume": 1.0
        },
        {
            "deinterlace_field_order": 0,
            "deinterlace_mode": 0,
            "enabled": true,
            "flags": 0,
            "hotkeys": {
                "OBSBasic.SelectScene": []
            },
            "id":  smallPicture.type === 'webcam' ? "av_capture_input" : "display_capture",
            "mixers": 0,
            "muted": false,
            "name": "Small Picture",
            "push-to-mute": false,
            "push-to-mute-delay": 0,
            "push-to-talk": false,
            "push-to-talk-delay": 0,
            "settings":this.obsSettingsFor(smallPicture),
            "sync": 0,
            "volume": 1.0
        }
    ]
  }

  saveLayout() {
    if (this.state.mainPicture) {
      fileUpdate('basic/scenes/Untitled.json',
        {
          sources: this.obsSources(
            this.state.mainPicture,
            this.state.smallPicture
          )
        }, 'json')
    }
  }

  renderVideoInputs(videos) {
    return videos.map((v) =>
      <Draggable
        axis="both"
        position={{ x: 0, y: 0 }}
        onStart={this.selectVideoInput.bind(this, v)}
        onStop={this.persistPreview.bind(this, v)}
        key={v.stream.id}
      >
        <video src={v.url} className={styles.videoStream} />
      </Draggable>
    );
  }

  render() {
    return (
      <div>
        <div className={styles.header}>
          <Link to="/step_1" className={styles.backKey}>
            <i className="fa fa-arrow-left fa-3x" />
          </Link>
          <Link to="/step_1" className={styles.forwardKey}>
            <i className="fa fa-arrow-right fa-3x" />
          </Link>
          <div className={styles.title}>
            <h2>{'Drag to Create Your Video Layout'}</h2>
          </div>
        </div>
        <div className={styles.container}>
          <div className={styles.left}>
            {this.renderVideoInputs(this.state.videoDevices)}
          </div>
          <div className={styles.right}>
            <div className={styles.picture}>
              <div
                className={styles.deleteMainPicture}
                onClick={this.deleteMainPicture.bind(this)}
              >X</div>
              <video
                className={styles.mainPicture}
                src={(this.state.mainPicturePreview && this.state.mainPicturePreview.url)
                      || (this.state.mainPicture && this.state.mainPicture.url) || ''}
                onMouseOver={this.hoverOverMainPicture.bind(this)}
                onMouseOut={this.leaveMainPicture.bind(this)}
              />
              <div
                className={styles.deleteSmallPicture}
                onClick={this.deleteSmallPicture.bind(this)}
              >X</div>
              <video
                className={styles.smallPicture}
                src={(this.state.smallPicturePreview && this.state.smallPicturePreview.url)
                      || (this.state.smallPicture && this.state.smallPicture.url) || ''}
                onMouseOver={this.hoverOverSmallPicture.bind(this)}
                onMouseOut={this.leaveSmallPicture.bind(this)}
              />
            </div>
          </div>
        </div>
      </div>
    );
  }
}
