import { exec } from 'child-process-promise';
import Installer from './installer';
import { fileRead } from '../utils/obs';

export default class MacInstaller extends Installer {
  install(setProgress, logger) {
    Promise.resolve()
    .then(() => logger('Downloading audio driver'))
    .then(this.ensureAudioDriverDownloaded.bind(this))
    .then((audioDriverPath) => {
      logger('Installing audio driver');
      return audioDriverPath;
    })
    .then((this.ensureAudioDriverInstalled.bind(this)))
    .then(() => setProgress(50))
    .then(() => logger('Downloading obs'))
    .then((this.ensureObsDownloaded.bind(this)))
    .then(() => logger('Installing OBS'))
    .then(this.ensureOBSInstalled.bind(this))
    .then(() => logger('OBS Installed'))
    .then(() => logger('Opening and Closing OBS to make sure everything is setup correctly'))
    .then(this.ensureOBSLicenseAccepted.bind(this))
    .then(() => setProgress(100))
    .catch((err) => console.log(err));
  }

  ensureAudioDriverDownloaded() {
    return this.downloadFile('https://s3-us-west-2.amazonaws.com/twitch-creative-assets/streamer_setup/audio_driver.dmg');
  }

  ensureObsDownloaded() {
    return this.downloadFile('https://s3-us-west-2.amazonaws.com/twitch-creative-assets/streamer_setup/obs_installer.dmg');
  }

  audioDriverRequired() {
    return Promise.resolve()
    .then(() =>
      navigator.mediaDevices.enumerateDevices().then((devices) =>
        !devices.some(device => device.label === 'iShowU Audio Capture')
      )
    );
  }

  ensureAudioDriverInstalled(audioDriverPath) {
    return Promise.resolve()
    .then(this.audioDriverRequired)
    .then((audioDriverRequired) => {
      if (!audioDriverRequired) return Promise.resolve();
      return Promise.resolve()
      .then(() => exec(`hdiutil attach ${audioDriverPath}`))
        .then(() => exec('open /Volumes/iShowU\\ Audio\\ Capture-1.0.1/iShowU\\ Audio\\ Capture.pkg -W'))
        .then(() => exec('hdiutil detach /Volumes/iShowU\\ Audio\\ Capture-1.0.1/iShowU\\ Audio\\ Capture.pkg'));
    })
    .catch(this.ensureAudioDriverInstalled.bind(this));
  }

  obsInstalled() {
    return Promise.resolve()
    .then(() => exec('[ -d  /Applications/OBS.app ]'))
    .then(() => true)
    .catch(() => false);
  }

  ensureOBSInstalled(obsPath) {
    return Promise.resolve()
    .then(this.obsInstalled)
    .then((obsInstalled) => {
      if (obsInstalled) return Promise.resolve();
      return Promise.resolve()
        .then(() => exec(`hdiutil attach ${obsPath}`))
        .then(() => exec('open /Volumes/OBS\\ Studio/OBS.pkg -W '))
        .then(() => exec('hdiutil detach /Volumes/OBS\\ Studio/OBS.pkg'));
    })
    .catch(this.ensureOBSInstalled.bind(this));
  }

  obsLicenseAccepted() {
    return Promise.resolve()
    .then(() => fileRead('global.ini', 'ini').General.LicenseAccepted)
  }

  ensureOBSLicenseAccepted() {
    return Promise.resolve()
    .then(this.obsLicenseAccepted)
    .then((obsLicenseAccepted) => {
      if (obsLicenseAccepted) return Promise.resolve();
      return Promise.resolve()
      .then(() => exec('open /Applications/OBS.app -g -W'))
      .then(this.obsLicenseAccepted)
      .then((confirmObsLicenseAccepted) => {
        if (!confirmObsLicenseAccepted) {
          return this.ensureOBSLicenseAccepted();
        }
        return Promise.resolve();
      });
    })
    .catch(this.ensureOBSLicenseAccepted.bind(this));
  }
}
