defmodule Outreachtool.InstagramScraper do
  import Ecto.Query, only: [from: 2]
  alias Outreachtool.{InstagramUser,InstagramPost,Repo}
  alias Experimental.Flow
  require ExAws.SQS

  def poll() do
      with {:ok, message} <- ExAws.request(ExAws.SQS.receive_message("instagram-user-scrape")),
           %{body: %{messages: [%{receipt_handle: receipt_handle}]}} <- message,
           parsed_message <- parse(message),
           :ok <- scrape(parsed_message),
           {:ok, _resp} <- ExAws.request(ExAws.SQS.delete_message("instagram-user-scrape", receipt_handle))
      do
        :ok
      else
        result -> result
      end
      poll()
  end

  def parse(payload) do
    message = List.first(payload.body.messages)
    message.body
  end

  def scrape([], _, _), do: :ok
  def scrape(user) do
    case usernames_to_scrape([user]) do
      [] -> :ok
      _ ->
        touch(user)
        posts = fetch_posts(user)
        |> Enum.map(&convert_to_db_post/1)
        |> Enum.take(100)

        usernames = posts
        |> Flow.from_enumerable
        |> Flow.partition(stages: 1)
        |> Flow.map(fn(%{:instagram_id => id}) -> id end)
        |> Flow.flat_map(&fetch_comments/1)
        |> Flow.map(fn(%{:username => username}) -> username end)
        |> Enum.uniq

        to_visit = usernames_to_scrape(usernames)
        SQS.send_in_batches(to_visit)
        Repo.insert_all(InstagramPost, posts, [on_conflict: :replace_all, conflict_target: :instagram_id])
        :ok
    end
  end

  def touch(username) do
    utc = :erlang.universaltime
    |> :calendar.datetime_to_gregorian_seconds
    |> :calendar.gregorian_seconds_to_datetime
    |> Ecto.DateTime.cast!

    InstagramUser.changeset(%InstagramUser{}, %{updated_at: utc, username: username})
    |> Repo.insert_or_update
  end

  def usernames_to_scrape(usernames) do
    one_day_ago =  NaiveDateTime.utc_now()
    |> NaiveDateTime.add(-1 * 60 * 60 * 24, :second)
    already_visited = Repo.all(from(u in InstagramUser, where: u.username in ^usernames and u.updated_at < ^one_day_ago))
    |> Enum.map(fn((%InstagramUser{username: username})) -> username end)
    usernames -- already_visited
  end

  def convert_to_db_post(post) do
    post
    |> Map.put(:instagram_id, post.id)
    |> Map.delete(:id)
    |> Map.put(:updated_at, Ecto.DateTime.utc)
    |> Map.put(:inserted_at, Ecto.DateTime.utc)
  end

  def fetch_comments(instagram_id), do: fetch("instagram-screen-scrape comments -p #{instagram_id}")
  def fetch_posts(user), do: fetch("instagram-screen-scrape posts -u #{user}")
  def fetch_user(user) do
    body = HTTPoison.get!("https://www.instagram.com/#{user}/").body
    Regex.run(~r/window._sharedData = (\{.*\});\<\/script\>/, body)
    |> List.last
    |> Poison.decode!
  end

  defp fetch(cmd) do
    [command|rest] = String.split(cmd) |> IO.inspect
    with {json, 0} <- System.cmd(command, rest),
      {:ok, objs} <- Poison.decode(json)
    do
      (for obj <- objs, do: for {key, val} <- obj, into: %{}, do: {String.to_atom(key), val})
    else
      _ -> []
    end
  end
end
