defmodule User do
  defstruct [
    :display_name,
    youtube_url: :not_found,
    youtube_subscribers: :not_found,
    instagram_url: :not_found,
    instagram_followers: :not_found,
    twitter_url: :not_found,
    twitter_followers: :not_found,
    patreon_url: :not_found,
    patreon_patrons: :not_found,
    instagram_url: :not_found,
    instagram_followers: :not_found,
    twitch_url: :not_found,
    twitch_followers: :not_found,
  ]

  def populate(%User{youtube_url: url, youtube_subscribers: :not_found} = user) when is_bitstring(url) do
    user
    |> Map.merge(get_youtube_info(user), &choose_the_longer/3)
    |> populate
  end

  def populate(%User{twitch_url: url, twitch_followers: :not_found} = user) when is_bitstring(url) do
    user
    |> Map.merge(get_twitch_info(user), &choose_the_longer/3)
    |> populate
  end

  def populate(%User{twitter_url: url, twitter_followers: :not_found} = user) when is_bitstring(url) do
    user
    |> Map.merge(get_twitter_info(user), &choose_the_longer/3)
    |> populate
  end

  def populate(%User{patreon_url: url, patreon_patrons: :not_found} = user) when is_bitstring(url) do
    user
    |> Map.merge(get_patreon_info(user), &choose_the_longer/3)
    |> populate
  end

  def populate(%User{instagram_url: url, instagram_followers: :not_found} = user) when is_bitstring(url) do
    user
    |> Map.merge(get_instagram_info(user), &choose_the_longer/3)
    |> populate
  end

  def populate(user), do: user

  def choose_the_longer(_k, v1, v2) when is_atom(v1), do: v2
  def choose_the_longer(_k, v1, v2) when is_atom(v2), do: v1
  def choose_the_longer(_k, v1, v2), do: Enum.max_by([v1, v2], &String.length/1)

  def get_youtube_info(user) do
    body = HTTPoison.get!(user.youtube_url <> "/about", %{}, [hackney: [{:follow_redirect, true}]]).body
    extract_links(body)
    |> Map.merge(%{youtube_subscribers: get_youtube_subscribers(body)})
  end

  def get_youtube_subscribers(html) do
    case Regex.run(~r/aria-label=\"([0-9\,]+) subscribers\"/, html) do
      nil -> nil
      result ->
        result
        |> List.last
        |> String.replace(",", "")
        |> Integer.parse
        |> elem(0)
      end
  end

  def get_youtube_subscribers(html) do
    case Regex.run(~r/aria-label=\"([0-9\,]+) subscribers\"/, html) do
      nil -> nil
      result ->
        result
        |> List.last
        |> String.replace(",", "")
        |> Integer.parse
        |> elem(0)
      end
  end

  def get_twitch_info(user) do
    username = String.split(user.twitch_url, "/") |> List.last
    get_panels(username)
    |> extract_links
    |> Map.merge(%{twitch_followers: get_twitch_followers(username)})
  end

  def get_panels(username) do
    HTTPoison.get!("https://api.twitch.tv/api/channels/#{username}/panels", %{"Client-ID" => "jzkbprff40iqj646a697cyrvl0zt2m6"}).body
  end

  def get_twitch_followers(username) do
    HTTPoison.get!("https://api.twitch.tv/kraken/channels/#{username}/follows", %{"Client-ID" => "jzkbprff40iqj646a697cyrvl0zt2m6"}).body
    |> Poison.decode!
    |> Map.get("_total")
  end

  def get_twitter_info(user) do
    body = HTTPoison.get!(user.twitter_url, %{}, [hackney: [{:follow_redirect, true}]]).body
    extract_links(body)
    |> Map.merge(%{twitter_followers: get_twitter_followers(body)})
  end

  def get_twitter_followers(html) do
    case Regex.run(~r/title=\"([0-9\,]+) Followers\"/, html) do
      nil -> nil
      result ->
        result
        |> List.last
        |> String.replace(",", "")
        |> Integer.parse
        |> elem(0)
      end
  end


  def get_patreon_info(user) do
    body = HTTPoison.get!(user.patreon_url, %{}, [hackney: [{:follow_redirect, true}]] ).body
    extract_links(body)
    |> Map.merge(%{patreon_patrons: get_patreon_patrons(body)})
  end

  def get_patreon_patrons(html) do
    case Regex.run(~r/"patron_count": ([0-9\,]+)/, html) do
      nil -> nil
      result ->
        result
        |> List.last
        |> String.replace(",", "")
        |> Integer.parse
        |> elem(0)
      end
  end

  def get_instagram_info(user) do
    body = HTTPoison.get!(user.instagram_url, %{}, [hackney: [{:follow_redirect, true}]] ).body
    extract_links(body)
    |> Map.merge(%{instagram_followers: get_instagram_followers(body)})
  end

  def get_instagram_followers(html) do
    case Regex.run(~r/\"followed_by\": {\"count\": ([0-9\,]+)}/, html) do
      nil -> nil
      result ->
        result
        |> List.last
        |> String.replace(",", "")
        |> Integer.parse
        |> elem(0)
      end
  end


  def extract_links(html) do
    %{}
    |> Map.merge(extract_instagram_url(html))
    |> Map.merge(extract_twitter_url(html))
    |> Map.merge(extract_facebook_url(html))
    |> Map.merge(extract_patreon_url(html))
    |> Map.merge(extract_twitch_url(html))
  end

  def extract_instagram_url(html), do: extract_url(html, :instagram_url, ~r/h?t?t?p?s?:?\/?\/?w?w?w?\.?instagram.com\/[a-zA-Z0-9_-]+/)
  def extract_twitter_url(html), do: extract_url(html, :twitter_url, ~r/h?t?t?p?s?:?\/?\/?w?w?w?\.?twitter.com\/[a-zA-Z0-9_-]+/)
  def extract_facebook_url(html), do: extract_url(html, :facebook_url, ~r/h?t?t?p?s?:?\/?\/?w?w?w?\.?facebook.com\/[a-zA-Z0-9_-]+/)
  def extract_patreon_url(html), do: extract_url(html, :patreon_url, ~r/h?t?t?p?s?:?\/?\/?w?w?w?\.?patreon.com\/[a-zA-Z0-9_-]+/)
  def extract_twitch_url(html), do: extract_url(html, :twitch_url, ~r/h?t?t?p?s?:?\/?\/?w?w?w?\.?twitch.tv\/[a-zA-Z0-9_-]+/)

  def extract_url(html, key, regex) do
    case Regex.run(regex, html) do
      nil -> %{}
      [url] -> %{key => url}
    end
  end
end
