package missioncontrol

import (
	"encoding/json"
	"fmt"
	"net/http"
	"strings"
	"time"

	"github.com/stvp/rollbar"

	"code.justin.tv/creative/streamer/lib/config"
	"code.justin.tv/creative/streamer/lib/util"
)

// StreamSessionVideo is the relationship between a stream session and a video
type StreamSessionVideo struct {
	ID              int       `json:"id"`
	StreamSessionID int       `json:"stream_session_id"`
	VideoID         int       `json:"video_id"`
	SecondsStreamed int       `json:"seconds_streamed"`
	Order           int       `json:"order"`
	State           string    `json:"state"`
	CreatedAt       time.Time `json:"created_at"`
	UpdatedAt       time.Time `json:"updated_at"`
	Video           struct {
		ID        int       `json:"id"`
		FileName  string    `json:"file_name"`
		Show      string    `json:"show"`
		Season    string    `json:"season"`
		S3Path    string    `json:"s3_path"`
		ChannelID int       `json:"channel_id"`
		Length    int       `json:"length"`
		CreatedAt time.Time `json:"created_at"`
		UpdatedAt time.Time `json:"updated_at"`
	} `json:"video"`
}

// StreamSession represents the full streaming session including all the videos
type StreamSession struct {
	ID                  int                  `json:"id"`
	ChannelID           int                  `json:"channel_id"`
	StartTime           time.Time            `json:"start_time"`
	State               string               `json:"state"`
	CreatedAt           time.Time            `json:"created_at"`
	UpdatedAt           time.Time            `json:"updated_at"`
	StreamSessionVideos []StreamSessionVideo `json:"stream_session_videos"`
	Channel             struct {
		ID        int       `json:"id"`
		Name      string    `json:"name"`
		StreamKey string    `json:"stream_key"`
		CreatedAt time.Time `json:"created_at"`
		UpdatedAt time.Time `json:"updated_at"`
	} `json:"channel"`
}

// StreamSessionResponse is the response from missioncontrol
type StreamSessionResponse struct {
	StreamSession StreamSession `json:"stream_session"`
}

// StreamSessionVideoResponse is the response from missioncontrol
type StreamSessionVideoResponse struct {
	StreamSessionVideo StreamSessionVideo `json:"stream_session_video"`
}

// StreamSessionVideoPutRequestPayload is the payload serializer
// TODO: switch to JSON
func StreamSessionVideoPutRequestPayload(state *string, secondsStreamed *int) string {
	payload := []string{}
	if state != nil {
		payload = append(payload, fmt.Sprintf("stream_session_video[state]=%s", *state))
	}
	if secondsStreamed != nil {
		payload = append(payload, fmt.Sprintf("stream_session_video[seconds_streamed]=%d", *secondsStreamed))
	}
	return strings.Join(payload, "&")
}

// UpdateStreamSessionVideo updates the streamsession video
func UpdateStreamSessionVideo(id *int, state *string, secondsStreamed *int) {
	payload := StreamSessionVideoPutRequestPayload(state, secondsStreamed)
	body := strings.NewReader(payload)
	req, err := http.NewRequest(
		"PUT",
		fmt.Sprintf("%s/stream_session_videos/%d", appconfig.Config.MissionControlURL, *id),
		body,
	)
	if err != nil {
		rollbar.Error(rollbar.ERR, err)
	}
	req.Header.Set("Content-Type", "application/x-www-form-urlencoded")
	resp, err := http.DefaultClient.Do(req)
	if err != nil {
		rollbar.Error(rollbar.ERR, err)
	}
	defer util.CloseAndReport(resp.Body)
}

// GetStreamSessionVideo updates the streamsession video
func GetStreamSessionVideo(ID *int) StreamSessionVideo {
	req, err := http.NewRequest(
		"GET",
		fmt.Sprintf("%s/stream_session_videos/%d", appconfig.Config.MissionControlURL, *ID),
		nil,
	)
	if err != nil {
		rollbar.Error(rollbar.ERR, err)
	}
	resp, err := http.DefaultClient.Do(req)
	if err != nil {
		rollbar.Error(rollbar.ERR, err)
	}
	defer util.CloseAndReport(resp.Body)
	ssvr := StreamSessionVideoResponse{}
	err = json.NewDecoder(resp.Body).Decode(&ssvr)
	if err != nil {
		rollbar.Error(rollbar.ERR, err)
	}
	return ssvr.StreamSessionVideo
}

// GetStreamSession updates the streamsession
func GetStreamSession(ID int) StreamSession {
	req, err := http.NewRequest(
		"GET",
		fmt.Sprintf("%s/stream_sessions/%d", appconfig.Config.MissionControlURL, ID),
		nil,
	)
	if err != nil {
		rollbar.Error(rollbar.ERR, err)
	}
	resp, err := http.DefaultClient.Do(req)
	if err != nil {
		rollbar.Error(rollbar.ERR, err)
	}
	defer util.CloseAndReport(resp.Body)
	ssr := StreamSessionResponse{}
	err = json.NewDecoder(resp.Body).Decode(&ssr)
	if err != nil {
		rollbar.Error(rollbar.ERR, err)
	}
	return ssr.StreamSession
}

// UpdateStreamSession updates the stream session
func UpdateStreamSession(id int, state string) {
	body := strings.NewReader(fmt.Sprintf("stream_session[state]=%s", state))
	req, err := http.NewRequest(
		"PUT",
		fmt.Sprintf("%s/stream_sessions/%d", appconfig.Config.MissionControlURL, id),
		body,
	)
	if err != nil {
		rollbar.Error(rollbar.ERR, err)
	}
	req.Header.Set("Content-Type", "application/x-www-form-urlencoded")
	resp, err := http.DefaultClient.Do(req)
	if err != nil {
		rollbar.Error(rollbar.ERR, err)
	}
	defer util.CloseAndReport(resp.Body)
}

// PopStreamSession pops a streamsession video off the queue.
func PopStreamSession(cooldown time.Duration) StreamSession {
	ssr := StreamSessionResponse{}
	for {
		resp, err := http.Post(
			fmt.Sprintf("%s/stream_sessions/pop", appconfig.Config.MissionControlURL),
			"application/json",
			nil,
		)
		if err != nil {
			rollbar.Error(rollbar.ERR, err)
		}
		if resp.StatusCode == 200 {
			err = json.NewDecoder(resp.Body).Decode(&ssr)
			if err != nil {
				rollbar.Error(rollbar.ERR, err)
			}
			return ssr.StreamSession
		}
		time.Sleep(cooldown)
	}
}
