package sessiontracker

import (
	"bufio"
	"errors"
	"regexp"
	"strconv"
	"strings"
	"time"

	"github.com/stvp/rollbar"

	"code.justin.tv/creative/streamer/lib/missioncontrol"
)

// ProcessFFMPEGLogs processes the logs and updates missionControl accordingly
func ProcessFFMPEGLogs(scanner *bufio.Scanner) {
	var currentSsvID int
	var doneChanPtr *chan bool

	for scanner.Scan() {
		text := scanner.Text()
		if strings.Contains(text, "GET /twitch-creative-video-repository") {
			ssv, err := getStreamSessionVideoFromFFMPEG(text)
			if err != nil {
				rollbar.Error(rollbar.ERR, err)
			} else {
				if currentSsvID != ssv.ID {
					if doneChanPtr != nil {
						*doneChanPtr <- true
					}
					doneChan := make(chan bool)
					doneChanPtr = &doneChan
					go trackStreamSessionVideo(ssv, &doneChan)
					currentSsvID = ssv.ID
				}
			}
		}
	}
	if doneChanPtr != nil {
		*doneChanPtr <- true
	}
}

func trackStreamSessionVideo(ssv *missioncontrol.StreamSessionVideo, doneChan *chan bool) {
	offset := float64(ssv.SecondsStreamed)
	secondsElapsed := offset
	ticker := time.NewTicker(1 * time.Second)
	startTime := time.Now()
	for {
		select {
		case <-*doneChan:
			ticker.Stop()
			return
		case timestamp := <-ticker.C:
			secondsElapsed = timestamp.Sub(startTime).Seconds()
			secondsElapsedInt := int(secondsElapsed)
			missioncontrol.UpdateStreamSessionVideo(&ssv.ID, nil, &secondsElapsedInt)
		}
	}
}

// getStreamSessionVideoFromFFMPEG extracts the current stream from ffmpegs logs.
// Unfortunately this is pretty noisy we should do some more complex logic to
// handle the edge case of two videos with the same name after each other
func getStreamSessionVideoFromFFMPEG(ffmpegLog string) (*missioncontrol.StreamSessionVideo, error) {
	re, err := regexp.Compile(`GET /twitch-creative-video-repository/(.*) HTTP/1.1`)
	if err != nil {
		return nil, err
	}
	res := re.FindAllStringSubmatch(ffmpegLog, -1)
	if len(res) > 0 && len(res[0]) > 0 {
		i, err := strconv.Atoi(res[0][1])
		if err != nil {
			return nil, err
		}
		ssv := missioncontrol.GetStreamSessionVideo(&i)

		return &ssv, nil
	}
	return nil, errors.New("StreamSessionVideo not found")
}
